/**
 * KareeRise Job Extension - UI Components
 * 
 * This module handles all UI-related functionality for the extension.
 */

// CSS Styles for the UI components - will be injected into the page
// Use a different approach to prevent duplicate declarations
let UI_STYLES;
if (typeof window.KareeRiseUIStyles === 'undefined') {
    window.KareeRiseUIStyles = true; // Mark as loaded
    UI_STYLES = `
.kareerise-container {
  position: fixed;
  right: 20px;
  top: 20%;
  z-index: 9999;
  display: flex;
  flex-direction: column;
  gap: 10px;
}

.kareerise-floating-button {
  display: flex;
  align-items: center;
  justify-content: center;
  gap: 6px;
  background-color: #2D7DE9;
  color: white;
  font-weight: 500;
  font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Arial, sans-serif;
  border: none;
  border-radius: 8px;
  padding: 10px 14px;
  font-size: 14px;
  cursor: pointer;
  transition: all 0.2s ease;
  box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
}

.kareerise-floating-button:hover {
  background-color: #4A90E2;
  box-shadow: 0 5px 15px rgba(0, 0, 0, 0.15);
  transform: translateY(-1px);
}

.kareerise-floating-button:active {
  transform: translateY(1px);
  box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
}

.kareerise-floating-button:disabled {
  background-color: #b0b0b0;
  cursor: not-allowed;
  transform: none;
}

.kareerise-button-icon {
  width: 16px;
  height: 16px;
}


.kareerise-notification {
  position: fixed;
  top: 20px;
  right: 20px;
  padding: 12px 16px;
  background-color: white;
  color: #333;
  border-radius: 8px;
  z-index: 10000;
  box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
  font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Arial, sans-serif;
  font-size: 14px;
  display: flex;
  align-items: center;
  animation: kareerise-notification-slide-in 0.3s ease-out forwards;
  max-width: 400px;
}

.kareerise-notification.success {
  border-left: 4px solid #4CAF50;
}

.kareerise-notification.error {
  border-left: 4px solid #F44336;
}

.kareerise-notification.info {
  border-left: 4px solid #2196F3;
}

.kareerise-notification-icon {
  margin-right: 8px;
  width: 20px;
  height: 20px;
}

.kareerise-notification-message {
  flex-grow: 1;
}

.kareerise-notification-close {
  background: none;
  border: none;
  color: #999;
  cursor: pointer;
  font-size: 18px;
  padding: 0;
  margin-left: 8px;
}

@keyframes kareerise-notification-slide-in {
  from {
    transform: translateX(100%);
    opacity: 0;
  }
  to {
    transform: translateX(0);
    opacity: 1;
  }
}

@keyframes kareerise-notification-slide-out {
  from {
    transform: translateX(0);
    opacity: 1;
  }
  to {
    transform: translateX(100%);
    opacity: 0;
  }
}

.kareerise-notification-exit {
  animation: kareerise-notification-slide-out 0.3s ease-in forwards;
}

.kareerise-tooltip {
  position: absolute;
  background-color: #333;
  color: white;
  padding: 5px 10px;
  border-radius: 4px;
  font-size: 12px;
  font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Arial, sans-serif;
  z-index: 10001;
  pointer-events: none;
  opacity: 0;
  transition: opacity 0.3s;
}
`;
} else {
    // If styles are already loaded, use a minimal fallback to avoid errors
    UI_STYLES = '/* Styles already loaded */';
}

// Add styles to the page
function injectStyles() {
    // Check if styles are already injected
    if (document.querySelector('style[data-kareerise-styles]')) {
        console.log('✅ KareeRise styles already injected, skipping');
        return;
    }
    
    const styleElement = document.createElement('style');
    styleElement.setAttribute('data-kareerise-styles', 'true');
    styleElement.textContent = UI_STYLES;
    document.head.appendChild(styleElement);
    console.log('✅ KareeRise styles injected');
}

// Create the floating buttons container
function createButtonContainer() {
    // Check if container already exists
    let container = document.querySelector('.kareerise-container');
    if (container) {
        return container;
    }
    
    container = document.createElement('div');
    container.className = 'kareerise-container';
    document.body.appendChild(container);
    
    return container;
}

// Create and add the floating "Add to KareeRise" button
function createAddButton() {
    const container = createButtonContainer();
    
    // Check if button already exists
    let button = container.querySelector('.kareerise-floating-button');
    if (button) {
        return button;
    }
    
    button = document.createElement('button');
    button.className = 'kareerise-floating-button';
    button.innerHTML = `
        <svg class="kareerise-button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
          <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z"/>
        </svg>
        Add to KareeRise
    `;
    button.title = 'Add this job to your KareeRise applications';
    container.appendChild(button);
    
    return button;
}

// Note: createAIButton function removed since we now only have one main button that uses AI extraction

// Show a notification message
function showNotification(message, type = 'info', duration = 5000) {
    // Remove any existing notifications
    const existingNotification = document.querySelector('.kareerise-notification');
    if (existingNotification) {
        existingNotification.remove();
    }
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `kareerise-notification ${type}`;
    
    // Add icon based on type
    let iconSvg = '';
    if (type === 'success') {
        iconSvg = '<svg class="kareerise-notification-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="#4CAF50"><path d="M9 16.2L4.8 12l-1.4 1.4L9 19 21 7l-1.4-1.4L9 16.2z"/></svg>';
    } else if (type === 'error') {
        iconSvg = '<svg class="kareerise-notification-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="#F44336"><path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/></svg>';
    } else {
        iconSvg = '<svg class="kareerise-notification-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="#2196F3"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z"/></svg>';
    }
    
    notification.innerHTML = `
        ${iconSvg}
        <span class="kareerise-notification-message">${message}</span>
        <button class="kareerise-notification-close">×</button>
    `;
    
    document.body.appendChild(notification);
    
    // Handle close button
    const closeButton = notification.querySelector('.kareerise-notification-close');
    closeButton.addEventListener('click', () => {
        notification.classList.add('kareerise-notification-exit');
        setTimeout(() => notification.remove(), 300);
    });
    
    // Auto-close after duration
    setTimeout(() => {
        if (document.body.contains(notification)) {
            notification.classList.add('kareerise-notification-exit');
            setTimeout(() => {
                if (document.body.contains(notification)) {
                    notification.remove();
                }
            }, 300);
        }
    }, duration);
    
    return notification;
}

// Update button state
function updateButtonState(button, state, message = null) {
    if (!button) return;
    
    switch(state) {
        case 'default':
            button.innerHTML = `
                <svg class="kareerise-button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z"/>
                </svg>
                Add to KareeRise
            `;
            button.disabled = false;
            button.style.backgroundColor = '#2D7DE9';
            break;
            
        case 'loading':
            button.innerHTML = `
                <svg class="kareerise-button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M12 6v3l4-4-4-4v3c-4.42 0-8 3.58-8 8 0 1.57.46 3.03 1.24 4.26L6.7 14.8c-.45-.83-.7-1.79-.7-2.8 0-3.31 2.69-6 6-6zm6.76 1.74L17.3 9.2c.44.84.7 1.79.7 2.8 0 3.31-2.69 6-6 6v-3l-4 4 4 4v-3c4.42 0 8-3.58 8-8 0-1.57-.46-3.03-1.24-4.26z"/>
                </svg>
                ${message || 'Processing with AI...'}
            `;
            button.disabled = true;
            button.style.backgroundColor = '#b0b0b0';
            break;
            
        case 'success':
            button.innerHTML = `
                <svg class="kareerise-button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M9 16.2L4.8 12l-1.4 1.4L9 19 21 7l-1.4-1.4L9 16.2z"/>
                </svg>
                ${message || 'Added Successfully'}
            `;
            button.disabled = true;
            button.style.backgroundColor = '#4CAF50';
            
            // Reset to default after 3 seconds
            setTimeout(() => {
                updateButtonState(button, 'default');
            }, 3000);
            break;
            
        case 'error':
            button.innerHTML = `
                <svg class="kareerise-button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/>
                </svg>
                ${message || 'Error'}
            `;
            button.disabled = false;
            button.style.backgroundColor = '#F44336';
            
            // Reset to default after 3 seconds
            setTimeout(() => {
                updateButtonState(button, 'default');
            }, 3000);
            break;
            
        case 'auth_required':
            button.innerHTML = `
                <svg class="kareerise-button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M11 7L9.6 8.4l2.6 2.6H2v2h10.2l-2.6 2.6L11 17l5-5-5-5zm9 12h-8v2h8c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2h-8v2h8v14z"/>
                </svg>
                Login Required
            `;
            button.disabled = false;
            button.style.backgroundColor = '#FF9800';
            break;
    }
}


// Expose UI functions to the window
window.KareeRiseUI = {
    injectStyles,
    createButtonContainer,
    createAddButton,
    showNotification,
    updateButtonState
};