/**
 * KareeRise Job Extension - Popup Script
 * 
 * This script handles the extension popup functionality
 */

// Server configuration from config.js
function getServerUrl() {
    return window.KareeRiseConfig?.SERVER_URL || "https://kareerise.com";
}

// DOM elements
let addJobBtn, openDashboardBtn, loginBtn;
let authIndicator, statusDot, authText;

// State
let currentTab = null;
let isJobPage = false;
let userAuthenticated = false;
let isProcessing = false;

// Initialize popup
document.addEventListener('DOMContentLoaded', async function() {
    console.log('🔧 KareeRise popup initializing...');
    
    // Get DOM elements
    addJobBtn = document.getElementById('add-job-btn');
    openDashboardBtn = document.getElementById('open-dashboard-btn');
    loginBtn = document.getElementById('login-btn');
    authIndicator = document.getElementById('auth-indicator');
    statusDot = document.getElementById('status-dot');
    authText = document.getElementById('auth-text');
    
    // Add event listeners
    addJobBtn.addEventListener('click', handleAddJob);
    openDashboardBtn.addEventListener('click', openDashboard);
    loginBtn.addEventListener('click', openLogin);
    
    // Initialize
    await initializePopup();
});

// Initialize popup state
async function initializePopup() {
    try {
        // Get current tab
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        currentTab = tab;
        
        console.log('Current tab:', tab.url);
        
        // Check if it's a job page
        isJobPage = checkIfJobPage(tab.url);
        
        // Check authentication
        await checkAuthentication();
        
        // Update UI state
        updateUIState();
        
    } catch (error) {
        console.error('Error initializing popup:', error);
        updateAuthStatus(false, 'Error');
    }
}

// Update authentication status display
function updateAuthStatus(isAuthenticated, message = '') {
    if (isAuthenticated) {
        statusDot.classList.remove('offline');
        authText.textContent = message || 'Logged in';
        loginBtn.style.display = 'none';
        addJobBtn.disabled = false;
    } else {
        statusDot.classList.add('offline');
        authText.textContent = message || 'Not logged in';
        loginBtn.style.display = 'block';
        addJobBtn.disabled = true;
    }
}

// Check if URL is a job page
function checkIfJobPage(url) {
    const lowerUrl = url.toLowerCase();
    
    // LinkedIn
    if (lowerUrl.includes('linkedin.com') && (lowerUrl.includes('/jobs/') || lowerUrl.includes('/job/'))) {
        return true;
    }
    
    // Indeed
    if (lowerUrl.includes('indeed.com') && 
        (lowerUrl.includes('/viewjob') || lowerUrl.includes('/job/') || lowerUrl.includes('jk=') || lowerUrl.includes('vjk='))) {
        return true;
    }
    
    // Glassdoor (all regions)
    if ((lowerUrl.includes('glassdoor.com') || lowerUrl.includes('glassdoor.sg') || lowerUrl.includes('glassdoor.ca') || lowerUrl.includes('glassdoor.co.uk')) && 
        (lowerUrl.includes('/job-listing/') || lowerUrl.includes('/jobs/') || lowerUrl.includes('/job/') || 
         lowerUrl.includes('/job_listing.htm') || lowerUrl.includes('joblistingid='))) {
        return true;
    }
    
    // ZipRecruiter
    if (lowerUrl.includes('ziprecruiter.com') && 
        (lowerUrl.includes('/jobs/') || lowerUrl.includes('/job/') || lowerUrl.includes('/c/') || 
         lowerUrl.includes('job_id=') || lowerUrl.includes('id='))) {
        return true;
    }
    
    // Generic job page patterns
    if ((lowerUrl.includes('/jobs/') || lowerUrl.includes('/job/') || lowerUrl.includes('/career/') || lowerUrl.includes('/careers/')) &&
        (lowerUrl.includes('position') || lowerUrl.includes('posting') || lowerUrl.includes('detail') || 
         lowerUrl.includes('view') || lowerUrl.includes('description'))) {
        return true;
    }
    
    return false;
}


// Check authentication status
async function checkAuthentication() {
    try {
        console.log('Checking authentication...');
        
        // First try using background script to get user info (has access to cookies)
        try {
            const backgroundResponse = await new Promise((resolve, reject) => {
                const timeout = setTimeout(() => {
                    reject(new Error('Background script timeout'));
                }, 8000);
                
                chrome.runtime.sendMessage(
                    { action: "getCurrentUser", url: getServerUrl() },
                    (response) => {
                        clearTimeout(timeout);
                        if (chrome.runtime.lastError) {
                            reject(new Error(chrome.runtime.lastError.message));
                        } else {
                            resolve(response);
                        }
                    }
                );
            });
            
            console.log('Background script auth response:', backgroundResponse);
            
            if (backgroundResponse && backgroundResponse.authenticated && backgroundResponse.userId) {
                userAuthenticated = true;
                updateAuthStatus(true, 'Logged in');
                return true;
            } else if (backgroundResponse && backgroundResponse.success === false) {
                console.log('Background script indicates not authenticated:', backgroundResponse.message);
            }
        } catch (backgroundError) {
            console.log('Background script auth failed, trying direct request:', backgroundError);
        }
        
        // Fallback to direct request with different credentials modes
        const credentialsModes = ['include', 'same-origin', 'omit'];
        
        for (const credentialsMode of credentialsModes) {
            try {
                console.log(`Trying auth check with credentials: ${credentialsMode}`);
                
                const response = await fetch(`${getServerUrl()}/api/user/current`, {
                    method: 'GET',
                    mode: 'cors',
                    credentials: credentialsMode,
                    headers: {
                        'Accept': 'application/json',
                        'X-Extension-Source': 'popup-auth-check'
                    },
                    signal: AbortSignal.timeout(5000)
                });
                
                console.log(`Auth response (${credentialsMode}):`, response.status, response.ok);
                
                if (response.ok) {
                    const data = await response.json();
                    console.log('Auth data:', data);
                    
                    if (data && data.user && data.user.id) {
                        userAuthenticated = true;
                        updateAuthStatus(true, 'Logged in');
                        return true;
                    }
                }
            } catch (requestError) {
                console.log(`Auth request failed with ${credentialsMode}:`, requestError.message);
            }
        }
        
        // Also try the test endpoint which might work better
        try {
            const testResponse = await fetch(`${getServerUrl()}/api/test-extension`, {
                method: 'GET',
                mode: 'cors',
                credentials: 'include',
                headers: {
                    'Accept': 'application/json',
                    'X-Extension-Source': 'popup-test-check'
                },
                signal: AbortSignal.timeout(5000)
            });
            
            if (testResponse.ok) {
                const testData = await testResponse.json();
                console.log('Test endpoint data:', testData);
                
                if (testData.hasSession && testData.session && testData.session.user) {
                    userAuthenticated = true;
                    updateAuthStatus(true, 'Logged in');
                    return true;
                }
            }
        } catch (testError) {
            console.log('Test endpoint failed:', testError.message);
        }
        
        userAuthenticated = false;
        updateAuthStatus(false, 'Not logged in');
        return false;
        
    } catch (error) {
        console.error('Auth check failed:', error);
        userAuthenticated = false;
        updateAuthStatus(false, 'Error');
        return false;
    }
}

// Update UI state based on current status
function updateUIState() {
    // Update button text based on page type
    if (isJobPage) {
        addJobBtn.innerHTML = `
            <svg class="button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z"/>
            </svg>
            Add This Job to KareeRise
        `;
    } else {
        addJobBtn.innerHTML = `
            <svg class="button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z"/>
            </svg>
            Extract Job from This Page
        `;
    }
}

// Handle add job button click
async function handleAddJob() {
    if (isProcessing) return;
    
    isProcessing = true;
    
    try {
        // Update button to loading state
        updateButtonState('loading');
        
        // Always inject scripts to ensure they're available - use try/catch to handle duplicates
        try {
            // Check if scripts are already loaded first
            let scriptsAlreadyLoaded = false;
            try {
                const results = await chrome.scripting.executeScript({
                    target: { tabId: currentTab.id },
                    function: () => {
                        return !!(window.KareeRiseAPIClient && window.KareeRiseExtractionHelpers && window.KareeRiseUI && window.KareeRiseConfig);
                    }
                });
                scriptsAlreadyLoaded = results[0].result;
            } catch (checkError) {
                console.log('⚠️ Could not check if scripts are loaded:', checkError.message);
            }
            
            // Only inject if not already loaded or if we're unsure
            if (!scriptsAlreadyLoaded) {
                console.log('🔄 Injecting content scripts...');
                await chrome.scripting.executeScript({
                    target: { tabId: currentTab.id },
                    files: ['js/config.js', 'js/api-client.js', 'js/extraction-helpers.js', 'js/ui-components.js', 'js/contact-extractor.js']
                });
                console.log('✅ Content scripts injected successfully');
            } else {
                console.log('✅ Content scripts already loaded, skipping injection');
            }
        } catch (injectionError) {
            console.log('⚠️ Content script injection failed:', injectionError.message);
            // Continue anyway - the page might have scripts from manifest
        }
        
        // Wait a moment for scripts to initialize
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // Execute content script to extract and add job
        const results = await chrome.scripting.executeScript({
            target: { tabId: currentTab.id },
            function: extractAndAddJob,
            args: [getServerUrl()]
        });
        
        const result = results[0].result;
        
        if (result.success) {
            updateButtonState('success');
            console.log('✅ Job extraction successful:', result.message);
            setTimeout(() => {
                updateButtonState('default');
            }, 3000);
        } else {
            console.error('❌ Job extraction failed:', result.error);
            updateButtonState('error');
            
            // Show more helpful error messages
            if (result.needsRefresh) {
                addJobBtn.innerHTML = `
                    <svg class="button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/>
                    </svg>
                    Refresh Page & Try Again
                `;
            } else if (result.needsLogin) {
                addJobBtn.innerHTML = `
                    <svg class="button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M11 7L9.6 8.4l2.6 2.6H2v2h10.2l-2.6 2.6L11 17l5-5-5-5zm9 12h-8v2h8c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2h-8v2h8v14z"/>
                    </svg>
                    Login Required
                `;
            }
            
            setTimeout(() => {
                updateButtonState('default');
            }, 5000);
        }
        
    } catch (error) {
        console.error('Error adding job:', error);
        
        // Handle extension context invalidation
        if (error.message && error.message.includes('Extension context invalidated')) {
            updateButtonState('error');
            addJobBtn.innerHTML = `
                <svg class="button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M12 6v3l4-4-4-4v3c-4.42 0-8 3.58-8 8 0 1.57.46 3.03 1.24 4.26L6.7 14.8c-.45-.83-.7-1.79-.7-2.8 0-3.31 2.69-6 6-6zm6.76 1.74L17.3 9.2c.44.84.7 1.79.7 2.8 0 3.31-2.69 6-6 6v-3l-4 4 4 4v-3c4.42 0 8-3.58 8-8 0-1.57-.46-3.03-1.24-4.26z"/>
                </svg>
                Extension Needs Reload
            `;
            
            // Add click handler to reload extension
            addJobBtn.onclick = () => {
                chrome.runtime.reload();
            };
            
            setTimeout(() => {
                addJobBtn.onclick = handleAddJob;
                updateButtonState('default');
            }, 8000);
        } else {
            updateButtonState('error');
            setTimeout(() => {
                updateButtonState('default');
            }, 3000);
        }
    } finally {
        isProcessing = false;
    }
}

// HTML extraction is now handled by extraction-helpers.js

// Helper function to try authentication via iframe (better cookie access)
async function tryIframeAuth(serverUrl) {
    return new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
            cleanup();
            reject(new Error('Iframe auth timeout'));
        }, 8000);
        
        const iframe = document.createElement('iframe');
        iframe.style.display = 'none';
        iframe.style.position = 'absolute';
        iframe.style.left = '-10000px';
        
        const cleanup = () => {
            clearTimeout(timeout);
            if (iframe.parentNode) {
                iframe.parentNode.removeChild(iframe);
            }
        };
        
        iframe.onload = () => {
            try {
                // Try to access iframe content and check auth
                fetch(`${serverUrl}/api/user/current`, {
                    method: 'GET',
                    credentials: 'include',
                    headers: {
                        'Accept': 'application/json',
                        'X-Extension-Source': 'iframe-auth'
                    }
                }).then(response => {
                    if (response.ok) {
                        return response.json();
                    }
                    throw new Error(`HTTP ${response.status}`);
                }).then(data => {
                    cleanup();
                    if (data && data.user && data.user.id) {
                        resolve({
                            userId: data.user.id,
                            email: data.user.email,
                            authenticated: true
                        });
                    } else {
                        resolve(null);
                    }
                }).catch(error => {
                    cleanup();
                    resolve(null); // Don't reject, just return null
                });
            } catch (error) {
                cleanup();
                resolve(null);
            }
        };
        
        iframe.onerror = () => {
            cleanup();
            resolve(null);
        };
        
        // Load the server's main page to establish session context
        iframe.src = `${serverUrl}/auth/login`;
        document.body.appendChild(iframe);
    });
}

// Content script function to extract and add job
async function extractAndAddJob(serverUrl = getServerUrl()) {
    try {
        console.log('🔄 Starting job extraction from popup...');
        
        // Wait for modules to be available if they were just injected
        let attempts = 0;
        const maxAttempts = 20;
        
        while ((!window.KareeRiseAPIClient || !window.KareeRiseExtractionHelpers) && attempts < maxAttempts) {
            await new Promise(resolve => setTimeout(resolve, 200));
            attempts++;
            console.log(`Waiting for extension modules... attempt ${attempts}`);
        }
        
        // Check if we have the required modules now
        if (window.KareeRiseAPIClient && window.KareeRiseExtractionHelpers && window.KareeRiseAPIClient.submitJobForAIExtraction) {
            console.log('✅ Using extension modules for extraction');
            
            try {
                // Extract page HTML using the proper module
                const htmlData = window.KareeRiseExtractionHelpers.extractPageHTML();
                console.log('📄 Extracted HTML data:', {
                    title: htmlData.title,
                    url: htmlData.url,
                    htmlLength: htmlData.fullHTML?.length || 0
                });
                
                // Get current user using the proper module
                const currentUser = await window.KareeRiseAPIClient.getCurrentUser();
                console.log('👤 Current user:', currentUser);
                
                if (!currentUser || !currentUser.authenticated) {
                    return { 
                        success: false, 
                        error: 'Authentication required. Please log in to KareeRise.',
                        needsLogin: true
                    };
                }
                
                // Submit for AI extraction using the proper module
                const result = await window.KareeRiseAPIClient.submitJobForAIExtraction(htmlData, currentUser);
                console.log('🎯 AI extraction result:', result);
                
                if (result.error) {
                    return { 
                        success: false, 
                        error: result.message || 'Failed to extract job information' 
                    };
                }
                
                return { 
                    success: true, 
                    message: result.message || 'Job added successfully!',
                    data: result
                };
                
            } catch (moduleError) {
                console.error('❌ Error using extension modules:', moduleError);
                // Fall through to direct API approach
            }
        }
        
        console.log('⚠️ Extension modules not available, using direct API approach...');
        
        // Fallback: Direct API approach when modules aren't available
        // Extract HTML directly - but optimized for smaller payload
        const htmlData = {
            title: document.title || 'Job Posting',
            url: window.location.href,
            fullHTML: window.KareeRiseExtractionHelpers ? 
                window.KareeRiseExtractionHelpers.extractOptimizedHTMLContent() : 
                document.body.innerHTML.substring(0, 25000),
            jobDescriptionHTML: document.querySelector('[class*="description"], [class*="job-details"], [id*="description"]')?.innerHTML || ''
        };
        
        console.log('📄 Fallback HTML extraction:', {
            title: htmlData.title,
            url: htmlData.url,
            htmlLength: htmlData.fullHTML?.length || 0
        });
        
        // Get authentication using a more robust approach
        let currentUser = null;
        
        // Method 1: Try to get current user through background script first (has better cookie access)
        try {
            console.log('🔐 Trying authentication via background script...');
            const bgResponse = await new Promise((resolve, reject) => {
                const timeout = setTimeout(() => reject(new Error('Background script timeout')), 3000);
                
                // Check if runtime is available before trying to send message
                if (!chrome.runtime || !chrome.runtime.sendMessage) {
                    reject(new Error('Chrome runtime not available'));
                    return;
                }
                
                try {
                    chrome.runtime.sendMessage(
                        { action: "getCurrentUser", url: serverUrl },
                        (response) => {
                            clearTimeout(timeout);
                            if (chrome.runtime.lastError) {
                                // Common context invalidation error
                                if (chrome.runtime.lastError.message.includes('Extension context invalidated')) {
                                    reject(new Error('Extension context invalidated - try refreshing the page'));
                                } else {
                                    reject(new Error(chrome.runtime.lastError.message));
                                }
                            } else {
                                resolve(response);
                            }
                        }
                    );
                } catch (sendError) {
                    clearTimeout(timeout);
                    reject(new Error(`Failed to send message: ${sendError.message}`));
                }
            });
            
            if (bgResponse && bgResponse.authenticated && bgResponse.userId) {
                currentUser = {
                    userId: bgResponse.userId,
                    email: bgResponse.email,
                    authenticated: true
                };
                console.log('✅ Got user via background script:', currentUser);
            }
        } catch (bgError) {
            console.log('⚠️ Background script auth failed:', bgError.message);
            // If it's a context invalidation, we should show a more helpful message
            if (bgError.message.includes('Extension context invalidated')) {
                return {
                    success: false,
                    error: 'Extension needs to be refreshed. Please reload the page and try again.',
                    needsRefresh: true
                };
            }
        }
        
        // Method 2: If background script failed, try direct fetch with different approaches
        if (!currentUser) {
            console.log('🔐 Trying direct authentication methods...');
            
            const authStrategies = [
                {
                    name: 'include-credentials',
                    options: {
                        method: 'GET',
                        mode: 'cors',
                        credentials: 'include',
                        headers: {
                            'Accept': 'application/json',
                            'X-Extension-Source': 'popup-direct-auth'
                        }
                    }
                },
                {
                    name: 'same-origin-credentials',
                    options: {
                        method: 'GET',
                        mode: 'cors',
                        credentials: 'same-origin',
                        headers: {
                            'Accept': 'application/json',
                            'X-Extension-Source': 'popup-same-origin-auth'
                        }
                    }
                }
            ];
            
            const authEndpoints = [
                `${serverUrl}/api/user/current`,
                `${serverUrl}/api/test-extension`
            ];
            
            for (const strategy of authStrategies) {
                if (currentUser) break;
                
                for (const endpoint of authEndpoints) {
                    try {
                        console.log(`🔐 Trying ${strategy.name} with ${endpoint}...`);
                        const authResponse = await fetch(endpoint, strategy.options);
                        
                        if (authResponse.ok) {
                            const authData = await authResponse.json();
                            console.log(`Auth data from ${endpoint}:`, authData);
                            
                            if (authData && authData.user && authData.user.id) {
                                currentUser = {
                                    userId: authData.user.id,
                                    email: authData.user.email,
                                    authenticated: true
                                };
                                console.log(`✅ Got user via ${strategy.name} + ${endpoint}:`, currentUser);
                                break;
                            } else if (authData.hasSession && authData.session && authData.session.user) {
                                currentUser = {
                                    userId: authData.session.user.id,
                                    email: authData.session.user.email,
                                    authenticated: true
                                };
                                console.log(`✅ Got user via ${strategy.name} test endpoint:`, currentUser);
                                break;
                            }
                        } else {
                            console.log(`❌ ${strategy.name} auth failed: ${authResponse.status}`);
                        }
                    } catch (authError) {
                        console.log(`❌ ${strategy.name} auth error with ${endpoint}:`, authError.message);
                    }
                }
            }
        }
        
        // Method 3: If direct methods failed, try iframe approach (sometimes better for cookies)
        if (!currentUser) {
            console.log('🔐 Trying iframe authentication method...');
            try {
                currentUser = await tryIframeAuth(serverUrl);
                if (currentUser) {
                    console.log('✅ Got user via iframe method:', currentUser);
                }
            } catch (iframeError) {
                console.log('⚠️ Iframe auth failed:', iframeError.message);
            }
        }
        
        if (!currentUser || !currentUser.authenticated) {
            return { 
                success: false, 
                error: 'Authentication required. Please log in to KareeRise.',
                needsLogin: true
            };
        }
        
        // Submit for AI extraction directly
        console.log('🤖 Submitting to AI extraction endpoint...');
        
        const extractionPayload = {
            pageUrl: htmlData.url,
            pageTitle: htmlData.title,
            htmlContent: htmlData.fullHTML,
            userId: currentUser.userId,
            user_id: currentUser.userId,
            userEmail: currentUser.email,
            timestamp: Date.now()
        };
        
        // Add timeout controller for AI extraction
        const controller = new AbortController();
        const timeoutId = setTimeout(() => {
            controller.abort();
            console.log('⌛ AI extraction timeout after 60 seconds');
        }, 60000); // 60 second timeout for AI processing
        
        const extractionResponse = await fetch(`${serverUrl}/api/extensions/ai-extract`, {
            method: 'POST',
            mode: 'cors',
            credentials: 'include',
            headers: {
                'Content-Type': 'application/json',
                'X-Extension-Source': 'popup-fallback-extraction',
                'X-Extension-Version': '1.2',
                'Accept': 'application/json',
                'X-Auth-User-ID': currentUser.userId,
                'X-User-ID': currentUser.userId
            },
            body: JSON.stringify(extractionPayload),
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        if (!extractionResponse.ok) {
            const errorText = await extractionResponse.text();
            throw new Error(`API call failed: ${extractionResponse.status} - ${errorText}`);
        }
        
        const result = await extractionResponse.json();
        console.log('🎯 Direct API extraction result:', result);
        
        if (result.error) {
            return { 
                success: false, 
                error: result.message || 'Failed to extract job information' 
            };
        }
        
        return { 
            success: true, 
            message: result.message || 'Job added successfully!',
            data: result
        };
        
    } catch (error) {
        console.error('❌ Error in extractAndAddJob:', error);
        return { 
            success: false, 
            error: error.message || 'An unexpected error occurred' 
        };
    }
}

// Update button visual state
function updateButtonState(state) {
    addJobBtn.className = 'main-button';
    
    switch(state) {
        case 'loading':
            addJobBtn.classList.add('loading');
            addJobBtn.disabled = true;
            addJobBtn.innerHTML = `
                <div class="spinner"></div>
                Processing with AI...
            `;
            break;
            
        case 'success':
            addJobBtn.classList.add('success');
            addJobBtn.disabled = true;
            addJobBtn.innerHTML = `
                <svg class="button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M9 16.2L4.8 12l-1.4 1.4L9 19 21 7l-1.4-1.4L9 16.2z"/>
                </svg>
                Added Successfully!
            `;
            break;
            
        case 'error':
            addJobBtn.classList.add('error');
            addJobBtn.disabled = false;
            addJobBtn.innerHTML = `
                <svg class="button-icon" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/>
                </svg>
                Failed - Try Again
            `;
            break;
            
        default: // 'default'
            addJobBtn.disabled = !userAuthenticated;
            updateUIState();
            break;
    }
}

// Open dashboard
function openDashboard() {
    chrome.tabs.create({ url: `${getServerUrl()}/applications` });
}

// Open login page
function openLogin() {
    chrome.tabs.create({ url: `${getServerUrl()}/auth/login?returnTo=/applications` });
}
