/**
 * KareeRise Job Extension - Main Content Script
 * 
 * This script runs on job listing pages and provides the functionality
 * to add jobs to KareeRise with a simple button click.
 */

// =============== STATE TRACKING ===============
// Track the current job URL and data to prevent duplicate submissions
let currentJobUrl = null;
let processingInProgress = false;
let extractingContacts = false;
let lastNavigationTime = Date.now();
let extractedContacts = [];

// =============== INITIALIZATION ===============
function init() {
    console.log("🚀 KareeRise Job Extension: Initializing...");
    console.log("🔍 Window modules check:");
    console.log("  - KareeRiseUI:", !!window.KareeRiseUI);
    console.log("  - KareeRiseAPIClient:", !!window.KareeRiseAPIClient);
    console.log("  - KareeRiseExtractionHelpers:", !!window.KareeRiseExtractionHelpers);
    
    // Ensure all modules are loaded before proceeding
    if (!window.KareeRiseUI || !window.KareeRiseAPIClient || !window.KareeRiseExtractionHelpers) {
        console.error("❌ Required modules not loaded yet. Retrying in 500ms...");
        setTimeout(init, 500);
        return;
    }
    
    console.log("✅ All modules loaded, proceeding with initialization");
    
    // Inject CSS styles
    window.KareeRiseUI.injectStyles();
    console.log("✅ CSS styles injected");
    
    // Add floating buttons if we're on a job page
    if (isJobPage()) {
        console.log("✅ Detected job page, setting up buttons");
        // Create the UI buttons
        setupUIButtons();
    } else {
        console.log("ℹ️ Not a job page, skipping button setup");
    }
    
    // Set up mutation observer to detect page changes
    setupPageObserver();
    console.log("✅ Page observer set up");
}

// =============== PAGE DETECTION ===============
// Check if we're on a job page
function isJobPage() {
    const url = window.location.href.toLowerCase();
    
    // LinkedIn job page detection
    if (url.includes('linkedin.com') && (url.includes('/jobs/') || url.includes('/job/'))) {
        return true;
    }
    
    // Indeed job page detection
    if (url.includes('indeed.com') && 
        (url.includes('/viewjob') || url.includes('/job/') || url.includes('jk=') || url.includes('vjk='))) {
        return true;
    }
    
    // Glassdoor job page detection (all regions)
    if ((url.includes('glassdoor.com') || url.includes('glassdoor.sg') || url.includes('glassdoor.ca') || url.includes('glassdoor.co.uk')) && 
        (url.includes('/job-listing/') || url.includes('/jobs/') || url.includes('/job/') || 
         url.includes('/job_listing.htm') || url.includes('jobListingId='))) {
        return true;
    }
    
    // ZipRecruiter job page detection
    if (url.includes('ziprecruiter.com') && 
        (url.includes('/jobs/') || url.includes('/job/') || url.includes('/c/') || 
         url.includes('job_id=') || url.includes('id='))) {
        return true;
    }
    
    // Generic job page detection by looking for common patterns
    if ((url.includes('/jobs/') || url.includes('/job/') || url.includes('/career/') || url.includes('/careers/')) &&
        (url.includes('position') || url.includes('posting') || url.includes('detail') || 
         url.includes('view') || url.includes('description'))) {
        return true;
    }
    
    return false;
}

// =============== UI SETUP ===============
// Set up UI buttons
function setupUIButtons() {
    // Create "Add to KareeRise" button (now using AI extraction)
    const addButton = window.KareeRiseUI.createAddButton();
    addButton.addEventListener('click', handleAIButtonClick); // Now uses AI extraction
}


// =============== EVENT HANDLERS ===============
// Note: The main "Add to KareeRise" button now uses AI extraction (handleAIButtonClick)
// The old direct extraction method (handleAddButtonClick) has been removed to simplify the UX

// Handle click on main "Add to KareeRise" button (now using AI extraction)
async function handleAIButtonClick(e) {
    e.preventDefault();
    
    // Ensure required modules are loaded
    if (!window.KareeRiseAPIClient || !window.KareeRiseExtractionHelpers || !window.KareeRiseUI) {
        console.error("❌ Required modules not loaded yet. Please try again in a moment.");
        alert("Extension modules not fully loaded. Please wait a moment and try again.");
        return;
    }
    
    // Prevent double-clicking
    if (processingInProgress) {
        console.log("🔄 Job processing already in progress, ignoring click");
        return;
    }
    
    processingInProgress = true;
    
    // Update button state
    const button = e.currentTarget;
    window.KareeRiseUI.updateButtonState(button, 'loading', 'Extracting job data...');
    
    try {
        // Show notification
        window.KareeRiseUI.showNotification("Extracting job information using AI...", "info");
        
        // Extract page HTML for AI
        let htmlData = null;
        try {
            htmlData = window.KareeRiseExtractionHelpers.extractPageHTML();
            console.log("🤖 Extracted HTML data for AI processing:", {
                title: htmlData.title,
                url: htmlData.url,
                contentLength: htmlData.fullHTML?.length || 0
            });
        } catch (extractionError) {
            console.error("❌ Error extracting HTML data:", extractionError);
            
            // Create minimal HTML data
            htmlData = {
                title: document.title || 'Job Posting',
                url: window.location.href,
                fullHTML: document.querySelector('main')?.innerHTML || document.body.innerHTML.substring(0, 50000),
                jobDescriptionHTML: document.querySelector('[class*="description"], [class*="job-details"], [id*="description"]')?.innerHTML || ''
            };
            
            console.log("⚠️ Using fallback HTML data");
        }
        
        // Get authentication info
        let currentUser = null;
        try {
            console.log("🔍 Checking authentication...");
            
            // First test if server is reachable at all
            console.log("🔄 Testing server connectivity...");
            try {
                const serverUrl = window.KareeRiseConfig?.SERVER_URL || 'https://kareerise.com';
                const testResponse = await fetch(`${serverUrl}/api/test-extension`, {
                    method: 'GET',
                    mode: 'cors',
                    credentials: 'include'
                });
                console.log("✅ Server connectivity test:", testResponse.status, testResponse.ok);
            } catch (connectError) {
                console.error("❌ Server connectivity test failed:", connectError);
                console.error("❌ This suggests the KareeRise server is not accessible");
            }
            
            currentUser = await window.KareeRiseAPIClient.getCurrentUser();
            console.log("🔍 Authentication result:", currentUser);
        } catch (authError) {
            console.warn('Authentication failed:', authError);
        }
        
        // Check if user is authenticated
        if (!currentUser || !currentUser.authenticated) {
            console.log("🔒 User not authenticated");
            window.KareeRiseUI.showNotification(
                "Please log in to KareeRise before adding jobs",
                "error"
            );
            
            // Open login page
            window.KareeRiseAPIClient.openLoginPage();
            
            // Update button state
            window.KareeRiseUI.updateButtonState(button, 'auth_required');
            return;
        }
        
        // Show notification for processing
        window.KareeRiseUI.showNotification("Analyzing job posting with AI...", "info");
        
        // Submit for AI extraction (GPT will extract contacts from HTML)
        const result = await window.KareeRiseAPIClient.submitJobForAIExtraction(
            htmlData,
            currentUser
        );
        
        // Handle the response
        if (result.error === "auth_required") {
            console.log("🔐 Authentication required");
            window.KareeRiseUI.showNotification(
                "Please log in to KareeRise before adding jobs",
                "error"
            );

            // Open login page in new tab
            window.KareeRiseAPIClient.openLoginPage();

            // Update button state
            window.KareeRiseUI.updateButtonState(button, 'auth_required');
        } else if (result.error) {
            console.error("❌ Error extracting job:", result.error);
            window.KareeRiseUI.showNotification(
                result.message || "Failed to extract job data. Please try again.",
                "error"
            );

            window.KareeRiseUI.updateButtonState(button, 'error', 'Failed');
        } else if (result.success && result.data) {
            // AI extraction succeeded - now create the application
            console.log("✅ Job data extracted:", result.data);
            window.KareeRiseUI.showNotification("Saving job to KareeRise...", "info");

            // Prepare job data from AI extraction result
            const jobData = {
                title: result.data.jobTitle || htmlData.title,
                company: result.data.companyName || 'Unknown Company',
                location: result.data.jobLocation || result.data.location || 'Unknown',
                description: result.data.jobDescription || result.data.description || '',
                jobUrl: result.data.jobUrl || htmlData.url,
                source: result.data.source || 'unknown',
                requirements: result.data.requirements || '',
                responsibilities: result.data.responsibilities || '',
                salary: result.data.salary || '',
                jobType: result.data.jobType || '',
                remote: result.data.remote || false,
                experience: result.data.experience || ''
            };

            // Extract contacts if present
            const contacts = result.data.contacts || [];

            console.log(`🔄 Submitting extracted job to create application (with ${contacts.length} contacts)`);

            // Submit the job application
            const submitResult = await window.KareeRiseAPIClient.submitJobApplication(
                jobData,
                contacts,
                currentUser
            );

            // Handle submission result
            if (submitResult.error) {
                console.error("❌ Error creating application:", submitResult.error);
                window.KareeRiseUI.showNotification(
                    submitResult.message || "Failed to save job. Please try again.",
                    "error"
                );
                window.KareeRiseUI.updateButtonState(button, 'error', 'Failed');
            } else {
                // Success! Application created
                console.log("✅ Application created successfully:", submitResult);
                window.KareeRiseUI.showNotification(
                    submitResult.message || "Job successfully added to KareeRise!",
                    "success"
                );
                window.KareeRiseUI.updateButtonState(button, 'success', 'Added!');
            }
        } else {
            // Unexpected response format
            console.error("❌ Unexpected response from AI extraction:", result);
            window.KareeRiseUI.showNotification(
                "Unexpected response format. Please try again.",
                "error"
            );
            window.KareeRiseUI.updateButtonState(button, 'error', 'Failed');
        }
    } catch (error) {
        console.error("❌ Error in add job handler:", error);
        window.KareeRiseUI.showNotification(
            `Error: ${error.message || "An unexpected error occurred"}`,
            "error"
        );
        
        window.KareeRiseUI.updateButtonState(button, 'error', 'Error');
    } finally {
        processingInProgress = false;
    }
}

// =============== PAGE OBSERVER ===============
// Set up mutation observer to detect job page changes
function setupPageObserver() {
    // Create observer to watch for URL or content changes
    const observer = new MutationObserver((mutations) => {
        // Check if significant time has passed since last navigation (200ms debounce)
        if (Date.now() - lastNavigationTime > 200) {
            lastNavigationTime = Date.now();
            
            // Only re-inject buttons if we're on a job page
            if (isJobPage()) {
                console.log("🔄 Page changed, re-injecting buttons");
                setupUIButtons();
            }
        }
    });
    
    // Start observing
    observer.observe(document.body, { 
        childList: true, 
        subtree: true 
    });
    
    // Also listen for URL changes on SPAs
    let lastUrl = window.location.href;
    setInterval(() => {
        if (window.location.href !== lastUrl) {
            lastUrl = window.location.href;
            lastNavigationTime = Date.now();
            
            // Only re-inject buttons if we're on a job page
            if (isJobPage()) {
                console.log("🔄 URL changed, re-injecting buttons");
                setupUIButtons();
            }
        }
    }, 500);
}

// =============== INITIALIZATION ===============
// Check if modules are loaded, and wait if needed
function checkAndInitialize() {
    if (window.KareeRiseAPIClient && window.KareeRiseExtractionHelpers && window.KareeRiseUI) {
        console.log("✅ All modules loaded, initializing extension");
        init();
    } else {
        console.log("⏳ Waiting for modules to load...");
        setTimeout(checkAndInitialize, 200);
    }
}

// Run initialization when the page is fully loaded
document.addEventListener('DOMContentLoaded', function() {
    console.log("🔄 KareeRise Job Extension: DOM Content Loaded");
    checkAndInitialize();
});

// Also run initialization now in case the page was already loaded
// This ensures we work properly with SPAs
checkAndInitialize();

// Log successful content script loading
console.log("📥 KareeRise Job Extension: Content script loaded");
console.log("📥 Current URL:", window.location.href);
console.log("📥 Is job page?", isJobPage());
