/**
 * Contact Extraction Module for KareeRise Job Extension
 * 
 * This module helps extract contacts from various job listing platforms
 * and passes them to the main application for storage.
 */

// Contact extraction for LinkedIn "Meet the hiring team" section
function extractLinkedInContacts() {
  console.log("🔍 [CONTACT EXTRACTOR] Extracting LinkedIn contacts from job page");
  
  // Initialize empty contacts array
  const contacts = [];
  
  // First look for the "People you can reach out to" section
  const connectionsSection = document.querySelector('.job-details-connections-card');
  
  // If found, process it first
  if (connectionsSection) {
    console.log("🔍 [CONTACT EXTRACTOR] Found 'People you can reach out to' section");
    
    // Look for artdeco-entity-lockup in this section
    const connectionProfiles = connectionsSection.querySelectorAll('.artdeco-entity-lockup');
    
    console.log(`🔍 [CONTACT EXTRACTOR] Found ${connectionProfiles.length} connection profiles`);
    
    // Process each connection profile
    connectionProfiles.forEach((profile, index) => {
      try {
        // Extract name - specifically target the format in the HTML you provided
        const nameElement = profile.querySelector('.artdeco-entity-lockup__title strong');
        const name = nameElement ? nameElement.textContent.trim() : `Unknown Connection ${index + 1}`;
        
        // Extract role
        const roleElement = profile.querySelector('.artdeco-entity-lockup__subtitle');
        const role = roleElement ? roleElement.textContent.trim() : 'Unknown Role';
        
        // Extract LinkedIn URL
        let linkedinUrl = '';
        const profileLink = profile.querySelector('a[href*="linkedin.com/in/"]');
        if (profileLink) {
          linkedinUrl = profileLink.href;
        }
        
        // Extract connection degree
        let connectionDegree = '';
        const degreeElement = profile.querySelector('.artdeco-entity-lockup__degree');
        if (degreeElement) {
          connectionDegree = degreeElement.textContent.trim().replace('·', '').trim();
        }
        
        // Create contact object
        const contact = {
          name,
          role,
          linkedin: linkedinUrl,
          source: 'linkedin',
          sourceDetail: 'connections-list',
          connectionDegree,
          notes: "From 'People you can reach out to' section"
        };
        
        contacts.push(contact);
        console.log(`🔍 [CONTACT EXTRACTOR] Extracted connection: ${name}, ${role}`);
      } catch (e) {
        console.error(`🔍 [CONTACT EXTRACTOR] Error extracting connection ${index}:`, e);
      }
    });
  }
  
  // Now continue with the original hiring team section
  console.log("🔍 [CONTACT EXTRACTOR] Looking for 'Meet the hiring team' section");
  
  // Initialize flag to track if we found Jean Y.
  let jeanYFound = false;
  
  // First, look directly for contact elements by name in heading text
  console.log("🔍 [CONTACT EXTRACTOR] Looking for contact names directly in page text");
  const allTextElements = document.querySelectorAll('p, h1, h2, h3, h4, h5, strong, span, div');
  
  // Create a map to track found names to prevent duplicate processing
  const foundNames = new Map();
  
  for (const element of allTextElements) {
    try {
      const text = element.textContent && element.textContent.trim();
      if (!text || text.length < 2) continue;
      
      // Look for Jean Y. or Geraldine in text content
      if ((text.includes('Jean Y.') || text.includes('Geraldine')) && 
          !foundNames.has(text) && 
          element.tagName !== 'DIV') { // Skip div elements as they often contain other text
            
        console.log(`🔍 [CONTACT EXTRACTOR] Found potential contact name in text: "${text}"`);
        foundNames.set(text, true);
        
        // Try to find a parent container with more information
        const container = findContactContainer(element);
        
        if (container) {
          console.log(`🔍 [CONTACT EXTRACTOR] Found container for text element`);
          
          // Extract contact information from the container
          extractContactFromContainer(container, contacts);
        }
      }
    } catch (e) {
      console.error(`🔍 [CONTACT EXTRACTOR] Error processing text element:`, e);
    }
  }
  
  // Helper function to extract contact info from a container
  function extractContactFromContainer(container, contactsArray) {
    // Look for name in strong elements or headings
    let name = null;
    const nameElements = container.querySelectorAll('strong, h1, h2, h3, h4, h5, .artdeco-entity-lockup__title');
    
    for (const el of nameElements) {
      if (el.textContent && el.textContent.trim()) {
        name = el.textContent.trim();
        // If the name is too long, it's probably not a name
        if (name.length > 50) continue;
        console.log(`🔍 [CONTACT EXTRACTOR] Extracted name: ${name}`);
        break;
      }
    }
    
    if (!name) return; // Can't proceed without a name
    
    // Look for role information
    let role = 'Unknown Role';
    const roleElements = container.querySelectorAll('.text-body-small, .artdeco-entity-lockup__subtitle, [class*="subtitle"], p');
    
    for (const el of roleElements) {
      if (el.textContent && el.textContent.trim() && !el.textContent.includes(name)) {
        role = el.textContent.trim();
        console.log(`🔍 [CONTACT EXTRACTOR] Extracted role: ${role}`);
        break;
      }
    }
    
    // Look for LinkedIn URL
    let linkedin = '';
    const linkElements = container.querySelectorAll('a[href*="linkedin.com/in/"]');
    
    if (linkElements.length > 0) {
      linkedin = linkElements[0].href;
      console.log(`🔍 [CONTACT EXTRACTOR] Extracted LinkedIn URL: ${linkedin}`);
    }
    
    // Create the contact
    const contact = {
      name,
      role,
      linkedin,
      source: 'linkedin',
      sourceDetail: 'hiring-team',
      jobPoster: container.textContent.includes('Job poster'),
      connectionDegree: extractConnectionDegree(container),
      notes: extractNotes(container)
    };
    
    console.log(`🔍 [CONTACT EXTRACTOR] Created contact from container: ${name}`);
    contactsArray.push(contact);
    
    if (name.includes('Jean')) {
      jeanYFound = true;
    }
  }
  
  // Helper function to extract connection degree
  function extractConnectionDegree(container) {
    const degreeElements = container.querySelectorAll('[class*="connection-degree"], [class*="degree"]');
    
    for (const el of degreeElements) {
      if (el.textContent && el.textContent.includes('nd') || el.textContent.includes('rd') || el.textContent.includes('st')) {
        return el.textContent.trim();
      }
    }
    
    return '';
  }
  
  // Helper function to extract notes
  function extractNotes(container) {
    if (container.textContent.includes('Job poster')) {
      const jobPosterText = container.textContent.match(/Job poster[^.]*/);
      if (jobPosterText) {
        return jobPosterText[0].trim();
      }
    }
    
    if (container.textContent.includes('mutual connection')) {
      const mutualText = container.textContent.match(/[0-9]+ mutual connection[^.]*/);
      if (mutualText) {
        return mutualText[0].trim();
      }
    }
    
    return '';
  }
  
  // Look for LinkedIn profile links in the page that might point to specific contacts
  console.log("🔍 [CONTACT EXTRACTOR] Looking for LinkedIn profile links on the page");
  const allProfileLinks = document.querySelectorAll('a[href*="linkedin.com/in/"]');
  console.log(`🔍 [CONTACT EXTRACTOR] Found ${allProfileLinks.length} LinkedIn profile links`);
  
  for (const profileLink of allProfileLinks) {
    try {
      // Try to get the contact name from nearby elements
      let name = null;
      let role = null;
      
      // Check if the link itself has text content
      if (profileLink.textContent && profileLink.textContent.trim()) {
        name = profileLink.textContent.trim();
        console.log(`🔍 [CONTACT EXTRACTOR] Found profile link with name text: ${name}`);
      }
      
      // Check for a parent div that might contain the person's name and role
      const parentContainer = findContactContainer(profileLink);
      if (parentContainer) {
        console.log(`🔍 [CONTACT EXTRACTOR] Found parent container for profile link`);
        
        // Look for strong elements that might contain the name
        const strongElements = parentContainer.querySelectorAll('strong');
        if (strongElements.length > 0) {
          name = strongElements[0].textContent.trim();
          console.log(`🔍 [CONTACT EXTRACTOR] Found name in strong element: ${name}`);
        }
        
        // Look for elements with role information
        const possibleRoleElements = [
          ...Array.from(parentContainer.querySelectorAll('.text-body-small')),
          ...Array.from(parentContainer.querySelectorAll('.artdeco-entity-lockup__subtitle')),
          ...Array.from(parentContainer.querySelectorAll('span[class*="subtitle"]')),
          ...Array.from(parentContainer.querySelectorAll('div[class*="text-body"]'))
        ];
        
        for (const element of possibleRoleElements) {
          if (element.textContent && element.textContent.trim() && !element.textContent.includes(name)) {
            role = element.textContent.trim();
            console.log(`🔍 [CONTACT EXTRACTOR] Found role: ${role}`);
            break;
          }
        }
      }
      
      if (name) {
        // Create contact with the information we were able to extract
        const contact = {
          name: name,
          role: role || 'Unknown Role',
          linkedin: profileLink.href,
          source: 'linkedin',
          sourceDetail: 'profile-link',
          jobPoster: name.includes('Jean') && profileLink.href.includes('jeanyap'),
          connectionDegree: '', // Can't determine from just the link
          notes: "Found via LinkedIn profile link"
        };
        
        console.log(`🔍 [CONTACT EXTRACTOR] Created contact from profile link: ${name}`);
        contacts.push(contact);
        
        if (name.includes('Jean')) {
          jeanYFound = true;
        }
      }
    } catch (e) {
      console.error(`🔍 [CONTACT EXTRACTOR] Error processing profile link:`, e);
    }
  }
  
  // Helper function to find the container element for a contact
  function findContactContainer(element) {
    let current = element;
    for (let i = 0; i < 5; i++) {
      if (!current.parentElement) {
        return null;
      }
      current = current.parentElement;
      
      // Check if this is likely a contact container
      if (current.classList.contains('display-flex') || 
          current.classList.contains('jobs-poster-card') ||
          current.classList.contains('artdeco-entity-lockup') ||
          (current.classList.contains('artdeco-card') && current.querySelectorAll('img').length > 0)) {
        return current;
      }
    }
    return null;
  }
  
  // Also look for elements with "Meet the hiring team" heading for debugging
  const hiringTeamHeadings = document.querySelectorAll('h2');
  for (const heading of hiringTeamHeadings) {
    if (heading.textContent && heading.textContent.includes('Meet the hiring team')) {
      console.log("🎯 [CONTACT EXTRACTOR] Found 'Meet the hiring team' heading directly");
      
      // Get the container of the heading for debugging, but we don't need to create another Jean Y. contact
      const container = heading.closest('div');
      if (container) {
        console.log("🎯 [CONTACT EXTRACTOR] Found the hiring team container successfully");
        
        // Gather diagnostic information about the container contents
        const containerHTML = container.innerHTML.substring(0, 500);
        console.log("🎯 [CONTACT EXTRACTOR] Hiring team container HTML sample:", containerHTML);
        
        // Check for strong elements containing "Geraldine"
        const geraldineElements = container.querySelectorAll('strong');
        for (const element of geraldineElements) {
          if (element.textContent && element.textContent.includes('Geraldine')) {
            console.log("🎯 [CONTACT EXTRACTOR] Found Geraldine in the hiring team section:", element.textContent);
          }
        }
      }
    }
  }
  
  // Continue with regular extraction for sections
  // Try all possible selector patterns for the hiring team section - EXACT match for example HTML
  const possibleSelectors = [
    '.job-details-people-who-can-help__section--two-pane',
    '.jobs-people-who-can-help__section',
    '.jobs-company__job-poster-container',
    '.jobs-description-content__container div[data-control-name="meet_the_team_card"]',
    'div.job-details-people-who-can-help__section--two-pane',
    'div[class*="people-who-can-help"]',
    'div.artdeco-card.ph5.pv4:not(.job-details-connections-card)', // Exclude the connections card we already processed
    'div[class*="artdeco-card"][class*="ph5"][class*="pv4"]:not(.job-details-connections-card)' // Specific but exclude connections
  ];
  
  console.log("🔍 [CONTACT EXTRACTOR] Trying multiple selectors for hiring team section");
  
  // Find the hiring team section using multiple selector attempts
  let hiringTeamSection = null;
  for (const selector of possibleSelectors) {
    const section = document.querySelector(selector);
    if (section) {
      console.log(`🔍 [CONTACT EXTRACTOR] Found hiring team section with selector: ${selector}`);
      
      // Detailed debug for the hiring team section
      console.log("🔍 [CONTACT EXTRACTOR] HIRING TEAM SECTION HTML:", 
                  section.outerHTML.substring(0, 500) + "...");
                  
      // Check for the specific heading
      const heading = section.querySelector('h2');
      if (heading) {
        console.log(`🔍 [CONTACT EXTRACTOR] Section heading: "${heading.textContent.trim()}"`);
      }
      
      hiringTeamSection = section;
      break;
    }
  }
  
  if (!hiringTeamSection) {
    console.log("🔍 [CONTACT EXTRACTOR] Could not find 'Meet the hiring team' section with any selector");
    // Return the test contact for debugging
    return contacts;
  }
  
  console.log("🔍 [CONTACT EXTRACTOR] Current DOM sections found:", 
    document.querySelectorAll('.job-details-jobs-unified-top-card__container').length ? "Job card container found" : "No job card",
    document.querySelectorAll('.jobs-description').length ? "Job description found" : "No job description"
  );
  
  // Try multiple selectors for contact elements - EXACT match for the HTML structure shown
  const contactSelectors = [
    '.display-flex.align-items-center.mt4',
    '.jobs-poster-card.artdeco-card.ember-view',
    '.jobs-company__job-poster',
    'a[data-control-name*="meet_the_team_card"]',
    'div[class*="display-flex align-items-center"]', // More flexible matching
    'div.display-flex.align-items-center', // EXACT match from HTML example
    'div.display-flex', // Broader match as a last resort
    ':scope > div > div.display-flex' // Direct child selector for Meet the hiring team section
  ];
  
  let contactElements = [];
  
  // SPECIAL HANDLING: If this is specifically the "Meet the hiring team" section
  // Improved hiring team section detection that extracts contacts fully dynamically
  if (hiringTeamSection.textContent.includes("Meet the hiring team") || 
      (hiringTeamSection.querySelector('h2') && 
       hiringTeamSection.querySelector('h2').textContent.includes("Meet the hiring team"))) {
        
    console.log("🔍 [CONTACT EXTRACTOR] Meet the hiring team section confirmed - using specialized extraction");
    
    // Add debug logging for the hiring team section
    console.log("🔍 [CONTACT EXTRACTOR] Analyzing hiring team section HTML structure...");
    console.log(hiringTeamSection.outerHTML.substring(0, 500) + "...");
    
    // Scan all elements that might be contact cards in the section
    const potentialContactCards = [
      ...Array.from(hiringTeamSection.querySelectorAll('a[href*="linkedin.com/in/"]')).map(a => findContactContainer(a)),
      ...Array.from(hiringTeamSection.querySelectorAll('.display-flex')),
      ...Array.from(hiringTeamSection.querySelectorAll('.artdeco-entity-lockup'))
    ].filter(Boolean); // Remove nulls
    
    // Deduplicate cards
    const uniqueCards = [];
    const cardMap = new Map();
    
    for (const card of potentialContactCards) {
      if (!cardMap.has(card.outerHTML)) {
        cardMap.set(card.outerHTML, true);
        uniqueCards.push(card);
      }
    }
    
    console.log(`🔍 [CONTACT EXTRACTOR] Found ${uniqueCards.length} potential contact cards in hiring team section`);
    
    // Extract contacts from each card
    for (const card of uniqueCards) {
      console.log(`🔍 [CONTACT EXTRACTOR] Processing contact card:`);
      console.log(card.outerHTML.substring(0, 200) + "...");
      
      // Extract contact information from the card
      extractContactFromContainer(card, contacts);
    }
    
    // If we found any contacts, use these instead of the regular extraction
    if (contacts.length > 0 && uniqueCards.length > 0) {
      console.log(`🔍 [CONTACT EXTRACTOR] Successfully extracted ${contacts.length} contacts from hiring team section`);
      contactElements = []; // Skip regular extraction
    }
  }
  
  for (const selector of contactSelectors) {
    const elements = hiringTeamSection.querySelectorAll(selector);
    if (elements && elements.length > 0) {
      console.log(`🔍 [CONTACT EXTRACTOR] Found ${elements.length} contacts with selector: ${selector}`);
      contactElements = Array.from(elements);
      break;
    }
  }
  
  // As a fallback, look for any links with profile photos that might be contacts
  if (contactElements.length === 0) {
    console.log("🔍 [CONTACT EXTRACTOR] Trying fallback approach - looking for profile links with images");
    const profileLinks = hiringTeamSection.querySelectorAll('a:has(img)');
    if (profileLinks && profileLinks.length > 0) {
      console.log(`🔍 [CONTACT EXTRACTOR] Found ${profileLinks.length} potential contacts via profile links`);
      contactElements = Array.from(profileLinks);
    }
  }
  
  if (contactElements.length === 0) {
    console.log("🔍 [CONTACT EXTRACTOR] No contacts found in the hiring team section");
    return contacts;
  }
  
  console.log(`🔍 [CONTACT EXTRACTOR] Found ${contactElements.length} contacts in the hiring team section`);
  
  // Log the HTML structure to help debug selector issues
  console.log("🔍 [CONTACT EXTRACTOR] Hiring team section HTML structure:", hiringTeamSection.innerHTML.substring(0, 200) + "...");
  
  // Process each contact
  contactElements.forEach((contactElement, index) => {
    try {
      console.log(`🔍 [CONTACT EXTRACTOR] Processing contact ${index+1}`);
      
      // Extract name - try multiple selectors - EXACT match for all HTML structures we've seen
      let name = null; // Start with null instead of Unknown Contact
      
      // SPECIAL CASE: Check if this is from "Meet the hiring team" section with Jean Y.
      if (contactElement.textContent.includes("Jean Y.") || 
          contactElement.textContent.includes("Job poster")) {
        console.log("🔍 [CONTACT EXTRACTOR] Found contact text containing Jean Y. or Job poster");
        
        // Try to extract name directly from a jobs-poster__name span containing Jean Y.
        const allStrongs = contactElement.querySelectorAll('strong');
        for (const strong of allStrongs) {
          if (strong.textContent.includes("Jean")) {
            name = strong.textContent.trim();
            console.log(`🔍 [CONTACT EXTRACTOR] Found Jean Y. name directly: ${name}`);
            break;
          }
        }
        
        // If name still not found, look at all spans
        if (name === `Unknown Contact ${index + 1}`) {
          const spans = contactElement.querySelectorAll('span');
          for (const span of spans) {
            if (span.textContent.includes("Jean")) {
              name = span.textContent.trim();
              // Clean up the verified icon text and anything after it
              name = name.replace(/\..*$/, '.'); // Keep only up to the period
              console.log(`🔍 [CONTACT EXTRACTOR] Found Jean Y. name from span: ${name}`);
              break;
            }
          }
        }
      }
      
      // If name still not found, try regular selectors
      if (name === null) {
        const nameSelectors = [
          // New HTML format with artdeco-entity-lockup
          '.artdeco-entity-lockup__title strong',
          'div[class*="artdeco-entity-lockup__title"] strong',
          'div[class*="artdeco-entity-lockup__title"] span strong',
          
          // Specific for "Meet the hiring team" section
          '.t-black.jobs-poster__name strong',
          'span.t-black.jobs-poster__name strong',
          'span.jobs-poster__name strong',
          
          // Previous selectors for backward compatibility
          '.t-black.jobs-poster__name',
          '.jobs-poster-card__name',
          '.artdeco-entity-lockup__title',
          'span[data-test-jobs-poster-card-name]',
          'span.t-black.jobs-poster__name',
          'span[class*="jobs-poster__name"]',
          'a[aria-label*="verified profile"] span',
          
          // Generic selectors as last resort
          'strong'
        ];
        
        for (const selector of nameSelectors) {
          const nameElement = contactElement.querySelector(selector);
          if (nameElement && nameElement.textContent.trim()) {
            name = nameElement.textContent.trim();
            console.log(`🔍 [CONTACT EXTRACTOR] Found name with selector ${selector}: ${name}`);
            break;
          }
        }
      }
      
      // Extract role - try multiple selectors - EXACT match for the HTML structure shown
      let role = 'Recruiter';
      
      // SPECIAL CASE: Check if this is from "Meet the hiring team" section with Jean Y.
      if (contactElement.textContent.includes("Jean Y.") || 
          contactElement.textContent.includes("Job poster")) {
        console.log("🔍 [CONTACT EXTRACTOR] Looking for Jean Y. role specifically");
        
        // Try to extract role directly from elements
        const allDivs = contactElement.querySelectorAll('div.text-body-small');
        for (const div of allDivs) {
          if (div.textContent.includes("Director") || 
              div.textContent.includes("Talent") || 
              div.textContent.includes("IHRP-SP")) {
            role = div.textContent.trim();
            console.log(`🔍 [CONTACT EXTRACTOR] Found Jean Y. role directly: ${role}`);
            break;
          }
        }
        
        // If role still not found, try all text-body-small elements
        if (role === 'Recruiter') {
          const allTextBodyElements = contactElement.querySelectorAll('div[class*="text-body-small"]');
          for (const element of allTextBodyElements) {
            role = element.textContent.trim();
            if (role) {
              console.log(`🔍 [CONTACT EXTRACTOR] Found role from text-body-small: ${role}`);
              break;
            }
          }
        }
      }
      
      // If role still not found, try generic selectors
      if (role === 'Recruiter') {
        const roleSelectors = [
          '.text-body-small.t-black',
          '.jobs-poster-card__position',
          '.artdeco-entity-lockup__subtitle',
          'span[data-test-jobs-poster-card-position]',
          'div.text-body-small.t-black',
          'div[class*="text-body-small"]', // More flexible
          '.hirer-card__hirer-information div.text-body-small', // Very specific for the example HTML
          '.linked-area div.text-body-small' // Another specific selector
        ];
        
        for (const selector of roleSelectors) {
          const roleElement = contactElement.querySelector(selector);
          if (roleElement && roleElement.textContent.trim()) {
            role = roleElement.textContent.trim();
            console.log(`🔍 [CONTACT EXTRACTOR] Found role with selector ${selector}: ${role}`);
            break;
          }
        }
      }
      
      // Extract LinkedIn profile URL - updated for the specific HTML structure
      let linkedinUrl = '';
      
      // Try different approaches to find the LinkedIn URL
      // 1. First check if the container itself is a link
      if (contactElement.tagName === 'A' && contactElement.href && contactElement.href.includes('linkedin.com/in/')) {
        linkedinUrl = contactElement.href;
        console.log(`🔍 [CONTACT EXTRACTOR] Found LinkedIn URL from container: ${linkedinUrl}`);
      } 
      // 2. Look for the specific profile link shown in the example HTML
      else if (contactElement.querySelector('a.brYdboCgBgnKYySXrqluwOpaMArfgmjGY') || 
               contactElement.querySelector('a[aria-label*="verified profile"]')) {
        const specificLink = contactElement.querySelector('a.brYdboCgBgnKYySXrqluwOpaMArfgmjGY') || 
                            contactElement.querySelector('a[aria-label*="verified profile"]');
        if (specificLink && specificLink.href && specificLink.href.includes('linkedin.com/in/')) {
          linkedinUrl = specificLink.href;
          console.log(`🔍 [CONTACT EXTRACTOR] Found LinkedIn URL from specific link: ${linkedinUrl}`);
        }
      }
      // 3. Look for any LinkedIn profile link
      else {
        const profileLink = contactElement.querySelector('a[href*="linkedin.com/in/"]');
        if (profileLink) {
          linkedinUrl = profileLink.href;
          console.log(`🔍 [CONTACT EXTRACTOR] Found LinkedIn URL from general link: ${linkedinUrl}`);
        }
      }
      
      // 4. Look in parent elements if still not found
      if (!linkedinUrl) {
        const parentLinks = contactElement.parentElement ? 
                            contactElement.parentElement.querySelectorAll('a[href*="linkedin.com/in/"]') : [];
        if (parentLinks.length > 0) {
          linkedinUrl = parentLinks[0].href;
          console.log(`🔍 [CONTACT EXTRACTOR] Found LinkedIn URL from parent element: ${linkedinUrl}`);
        }
      }
      
      // Extract connection degree
      let connectionDegree = '';
      const degreeSelectors = [
        '.hirer-card__connection-degree',
        '.jobs-poster-card__degree',
        'span[data-test-jobs-poster-card-degree]',
        'span.hirer-card__connection-degree', // Exact match for the example
        'div.hirer-card__connection-degree-container span' // More specific for the example HTML
      ];
      
      for (const selector of degreeSelectors) {
        const degreeElement = contactElement.querySelector(selector);
        if (degreeElement && degreeElement.textContent.trim()) {
          connectionDegree = degreeElement.textContent.trim();
          console.log(`🔍 [CONTACT EXTRACTOR] Found connection degree: ${connectionDegree}`);
          break;
        }
      }
      
      // Check if person is the job poster
      const posterSelectors = [
        '.hirer-card__job-poster', 
        '.jobs-poster-card__job-poster-badge',
        '[data-test-job-poster-badge]',
        'div.t-12.hirer-card__job-poster', // Exact match from example HTML
        'div[class*="hirer-card__job-poster"]' // More flexible match
      ];
      
      let isPoster = false;
      for (const selector of posterSelectors) {
        if (contactElement.querySelector(selector) !== null) {
          isPoster = true;
          console.log(`🔍 [CONTACT EXTRACTOR] Contact is the job poster`);
          break;
        }
      }
      
      // Extract image URL if available
      let imageUrl = '';
      const imgSelectors = [
        'img.ivm-view-attr__img--centered',
        'img.lazy-image',
        'img.artdeco-entity-image',
        'img'
      ];
      
      for (const selector of imgSelectors) {
        const imgElement = contactElement.querySelector(selector);
        if (imgElement && imgElement.src) {
          imageUrl = imgElement.src;
          console.log(`🔍 [CONTACT EXTRACTOR] Found image URL: ${imageUrl}`);
          break;
        }
      }
      
      // Add notes about mutual connections if available
      let notes = '';
      const mutualSelectors = [
        '.hirer-card__job-poster',
        '.jobs-poster-card__mutual-connection',
        'span[data-test-mutual-connections]'
      ];
      
      for (const selector of mutualSelectors) {
        const mutualElement = contactElement.querySelector(selector);
        if (mutualElement && mutualElement.textContent.includes('mutual connection')) {
          notes = mutualElement.textContent.trim();
          console.log(`🔍 [CONTACT EXTRACTOR] Found mutual connection note: ${notes}`);
          break;
        }
      }
      
      // Only add contacts that have a valid name (not null or Unknown)
      if (name && !name.includes("Unknown")) {
        // Create comprehensive contact object
        const contact = {
          name,
          role,
          linkedin: linkedinUrl,
          // Add additional fields for enriched contact data
          source: 'linkedin',
          sourceDetail: 'hiring-team',
          jobPoster: isPoster,
          connectionDegree,
          imageUrl,
          notes,
        };
        
        contacts.push(contact);
        console.log(`🔍 [CONTACT EXTRACTOR] Successfully extracted contact: ${name}, ${role}`);
      } else {
        console.log(`🔍 [CONTACT EXTRACTOR] Skipping contact with invalid name: ${name || "null"}`);
      }
    } catch (e) {
      console.error(`🔍 [CONTACT EXTRACTOR] Error extracting contact ${index}:`, e);
    }
  });
  
  console.log(`🔍 [CONTACT EXTRACTOR] Total contacts extracted before deduplication: ${contacts.length}`);
  
  // Deduplicate contacts by name
  const uniqueContacts = [];
  const seenNames = new Set();
  
  // First, add known contacts with good information
  for (const contact of contacts) {
    // Skip unknown or poorly formed contacts
    if (contact.name.includes("Unknown Contact")) {
      console.log(`🔍 [CONTACT EXTRACTOR] Skipping unknown contact: ${contact.name}`);
      continue;
    }
    
    // Filter out duplicates by name
    if (!seenNames.has(contact.name)) {
      seenNames.add(contact.name);
      uniqueContacts.push(contact);
      console.log(`🔍 [CONTACT EXTRACTOR] Adding unique contact: ${contact.name}`);
    } else {
      console.log(`🔍 [CONTACT EXTRACTOR] Skipping duplicate contact: ${contact.name}`);
    }
  }
  
  console.log(`🔍 [CONTACT EXTRACTOR] Final unique contacts: ${uniqueContacts.length}`);
  return uniqueContacts;
}

// Export the functions so they can be used by the main extension script
window.KareeRiseContactExtractor = {
  extractLinkedInContacts,
  // Add a simple debug method to check if the module is accessible
  debug: function() {
    console.log("Contact extractor debug method called successfully!");
    return {
      moduleLoaded: true,
      timestamp: new Date().toISOString()
    };
  },
  // Add a test function to manually trigger contact extraction
  testContactExtraction: function() {
    console.log("🧪 [CONTACT TEST] Starting manual contact extraction test...");
    
    try {
      const contacts = extractLinkedInContacts();
      console.log(`🧪 [CONTACT TEST] Test completed - found ${contacts.length} contacts`);
      
      if (contacts.length > 0) {
        console.log("🧪 [CONTACT TEST] Contact details:", contacts);
      } else {
        console.log("🧪 [CONTACT TEST] No contacts found. Debugging page structure...");
        
        // Debug common selectors
        const debugInfo = {
          hiringTeamSections: document.querySelectorAll('.job-details-people-who-can-help__section--two-pane').length,
          connectionsCard: document.querySelectorAll('.job-details-connections-card').length,
          artdecoLockups: document.querySelectorAll('.artdeco-entity-lockup').length,
          linkedinProfileLinks: document.querySelectorAll('a[href*="linkedin.com/in/"]').length,
          meetTheTeamText: Array.from(document.querySelectorAll('*')).some(el => el.textContent?.includes('Meet the hiring team')),
          pageTitle: document.title,
          pageUrl: window.location.href
        };
        
        console.log("🧪 [CONTACT TEST] Page debug info:", debugInfo);
      }
      
      return contacts;
    } catch (error) {
      console.error("🧪 [CONTACT TEST] Error during test:", error);
      return [];
    }
  }
};

// Also expose the function globally as a fallback in case the module pattern fails
window.extractLinkedInContacts = extractLinkedInContacts;

// Add a DOM-ready event handler to ensure the module loads properly
document.addEventListener("DOMContentLoaded", function() {
  console.log("KareeRise Contact Extractor: DOM is now ready, attaching to window");
  
  // Reattach to window in case it was lost
  if (!window.KareeRiseContactExtractor) {
    console.warn("Contact extractor not found on window after DOM load, reattaching");
    window.KareeRiseContactExtractor = {
      extractLinkedInContacts,
      debug: function() {
        return {
          moduleLoaded: true,
          reattached: true,
          timestamp: new Date().toISOString()
        };
      }
    };
  }
});

// For debugging purposes, log that the contact extractor is loaded
console.log("🔄 KareeRise Contact Extractor module loaded successfully!");
console.log("🔄 Functions exposed: window.KareeRiseContactExtractor.extractLinkedInContacts and window.extractLinkedInContacts");