// Background script for KareeRise Job Importer

// Listen for installation
chrome.runtime.onInstalled.addListener(() => {
  console.log('KareeRise Job Importer installed');
});

// Listen for startup to reinitialize
chrome.runtime.onStartup.addListener(() => {
  console.log('KareeRise Job Importer starting up');
  // Clear any stale cached data on startup
  currentUserId = null;
  currentUserEmail = null;
  lastAuthCheck = 0;
});

// Configure cookie domains - ensure these match your server settings
const COOKIE_DOMAINS = [
  'kareerise.com',
  'www.kareerise.com',
  'app.kareerise.com',
  'localhost',
  '127.0.0.1'
];

// Keep track of the current user
let currentUserId = null;
let currentUserEmail = null;
let lastAuthCheck = 0;
const AUTH_CHECK_INTERVAL = 5 * 60 * 1000; // 5 minutes


const DEFAULT_SERVER_URL = "https://kareerise.com";
const SERVER_URL = DEFAULT_SERVER_URL;
const DEFAULT_DOMAIN = (() => {
  try {
    return new URL(SERVER_URL).hostname;
  } catch {
    return 'kareerise.com';
  }
})();

// Handle messages from content script
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Background script received message:', message);
  
  // Handle cookie requests
  if (message.action === "getCookies") {
    const url = message.url || SERVER_URL;
    console.log('Getting cookies for URL:', url);
    
    // Create a URL object to get the domain
    let domain = DEFAULT_DOMAIN;
    
    try {
      if (url) {
        const urlObj = new URL(url);
        domain = urlObj.hostname;
      }
    } catch (e) {
      console.warn('Invalid URL format:', url);
    }
    
    // Get all domains we want to check for cookies
    const domainsToCheck = [domain, ...COOKIE_DOMAINS];
    const uniqueDomains = [...new Set(domainsToCheck)];
    
    console.log('Checking for cookies on domains:', uniqueDomains);
    
    // Collect all cookies from all relevant domains
    const cookiePromises = uniqueDomains.map(domainToCheck => {
      return new Promise(resolve => {
        chrome.cookies.getAll({ domain: domainToCheck }, cookies => {
          console.log(`Found ${cookies.length} cookies for domain: ${domainToCheck}`);
          resolve(cookies);
        });
      });
    });
    
    // When all cookie promises are resolved
    Promise.all(cookiePromises)
      .then(cookieArrays => {
        // Merge all cookie arrays into one, removing duplicates by name
        const allCookies = [];
        const cookieNames = new Set();
        
        cookieArrays.flat().forEach(cookie => {
          if (!cookieNames.has(cookie.name)) {
            cookieNames.add(cookie.name);
            allCookies.push(cookie);
          }
        });
        
        console.log(`Returning ${allCookies.length} total cookies to content script`);
        
        // Send the cookies back to the content script
        sendResponse({ 
          success: true, 
          cookies: allCookies,
          count: allCookies.length
        });
      })
      .catch(error => {
        console.error('Error getting cookies:', error);
        sendResponse({ 
          success: false, 
          cookies: [],
          error: error.toString()
        });
      });
    
    // Return true to indicate we'll respond asynchronously
    return true;
  }
  
  // Handle clear auth cache requests
  if (message.action === "clearAuthCache") {
    console.log("📋 Background: Clearing authentication cache");
    currentUserId = null;
    currentUserEmail = null;
    lastAuthCheck = 0;
    
    sendResponse({
      success: true,
      message: "Authentication cache cleared"
    });
    return true;
  }
  
  // Handle current user requests
  if (message.action === "getCurrentUser") {
    const url = message.url || SERVER_URL;
    const now = Date.now();
    
    console.log("📋 Background: getCurrentUser request for URL:", url);
    
    // No development mode fallback - always check with server
    
    // Skip cache for now to always get fresh auth data (this helps with account switching)
    // TODO: Re-enable caching after implementing proper cache invalidation
    // if (currentUserId && now - lastAuthCheck < AUTH_CHECK_INTERVAL) {
    //   // Using cached authentication
    //   sendResponse({
    //     success: true,
    //     userId: currentUserId,
    //     email: currentUserEmail,
    //     authenticated: true,
    //     cached: true
    //   });
    //   return true;
    // }
    
    // Otherwise check authentication with the server
    console.log(`Checking authentication with server at ${url}`);
    
    // Get cookies from all relevant domains
    const domainsToCheck = [new URL(url).hostname, ...COOKIE_DOMAINS];
    const uniqueDomains = [...new Set(domainsToCheck)];
    
    // Collect all cookies from all domains
    const cookiePromises = uniqueDomains.map(domainToCheck => {
      return new Promise(resolve => {
        chrome.cookies.getAll({ domain: domainToCheck }, cookies => {
          console.log(`Found ${cookies.length} cookies for domain: ${domainToCheck}`);
          resolve(cookies);
        });
      });
    });
    
    Promise.all(cookiePromises)
      .then(cookieArrays => {
        // Merge all cookie arrays and remove duplicates
        const allCookies = [];
        const cookieNames = new Set();
        
        cookieArrays.flat().forEach(cookie => {
          if (!cookieNames.has(cookie.name)) {
            cookieNames.add(cookie.name);
            allCookies.push(cookie);
          }
        });
        
        console.log(`Using ${allCookies.length} cookies for authentication`);
        
        // Create cookie header string
        const cookieString = allCookies.map(cookie => `${cookie.name}=${cookie.value}`).join('; ');
        
        console.log(`Using ${allCookies.length} cookies for authentication`);
        console.log('Cookie names:', allCookies.map(c => c.name));
        
        if (allCookies.length === 0) {
          console.log(`❌ No cookies found for ${DEFAULT_DOMAIN} - user may not be logged in`);
        }
        
        // Try multiple endpoints for authentication
        const apiUrls = [
          `${url}/api/test-extension`,
          `${url}/api/auth/session`
        ];
        
        // Try each endpoint
        const fetchPromises = apiUrls.map(apiUrl => {
          return fetch(apiUrl, {
            method: "GET",
            credentials: "include",
            headers: {
              "Content-Type": "application/json",
              "X-Extension-Source": "job-importer-extension-background",
              "X-Extension-Version": "1.2",
              "Cookie": cookieString,
              "X-Requested-With": "XMLHttpRequest",
              "Accept": "application/json, text/plain, */*"
            }
          })
          .then(response => {
            return { 
              url: apiUrl, 
              response,
              status: response.status,
              ok: response.ok
            };
          })
          .catch(error => {
            return { 
              url: apiUrl, 
              error: error.toString() 
            };
          });
        });
        
        // Execute all fetch requests in parallel
        return Promise.all(fetchPromises);
      })
      .then(results => {
        // Find the first successful response
        const successfulResponse = results.find(result => result.ok);
        
        if (successfulResponse) {
          return successfulResponse.response.json()
            .then(data => {
              return { 
                ...data,
                success: true 
              };
            });
        }
        
        // If no successful response, return authentication failure with details
        console.warn('All authentication endpoints failed:', results);
        return {
          authenticated: false,
          success: false,
          errors: results.map(r => ({
            url: r.url,
            status: r.status,
            error: r.error
          }))
        };
      })
      .then(data => {
        // Auth check completed
        console.log('Auth response:', data);
        
        // Store the user info for future use  
        // Handle test endpoint response
        if (data.hasSession && data.session && data.session.user) {
          console.log(`Successfully authenticated via test endpoint as user ${data.session.user.id}`);
          currentUserId = data.session.user.id;
          currentUserEmail = data.session.user.email;
          lastAuthCheck = now;
          
          sendResponse({
            success: true,
            userId: currentUserId,
            email: currentUserEmail,
            authenticated: true,
            fresh: true,
            method: 'test-endpoint'
          });
        } 
        // Handle NextAuth session response
        else if (data.user && data.user.id) {
          console.log(`Successfully authenticated via session endpoint as user ${data.user.id}`);
          currentUserId = data.user.id;
          currentUserEmail = data.user.email;
          lastAuthCheck = now;
          
          sendResponse({
            success: true,
            userId: currentUserId,
            email: currentUserEmail,
            authenticated: true,
            fresh: true,
            method: 'session-endpoint'
          });
        } else {
          // User not authenticated - clear any cached data
          console.log('Not authenticated:', data.error || 'No auth data');
          console.log('Clearing cached authentication data');
          currentUserId = null;
          currentUserEmail = null;
          lastAuthCheck = 0;
          
          sendResponse({
            success: false,
            authenticated: false,
            message: "Not authenticated",
            loginRequired: true
          });
        }
      })
      .catch(error => {
        console.error("Error checking authentication:", error);
        
        // Try to generate a more helpful error message
        let errorMsg = error.message || "Unknown error";
        
        if (errorMsg.includes('Failed to fetch') || errorMsg.includes('NetworkError')) {
          errorMsg = `Network error connecting to ${url}. Make sure the server is running and accessible.`;
        }
        
        if (errorMsg.includes('CORS')) {
          errorMsg = `CORS error - The server at ${url} is not configured to accept requests from the extension.`;  
        }
        
        console.log('Returning error response:', errorMsg);
        
        sendResponse({
          success: false,
          error: errorMsg,
          authenticated: false,
          url: url
        });
      });
    
    // Return true to indicate we'll respond asynchronously
    return true;
  }
  
  // Handle API proxy requests (to bypass CSP on pages like LinkedIn)
  if (message.action === "proxyFetch") {
    console.log('📡 Background: Proxying fetch request to:', message.url);

    const fetchOptions = {
      method: message.method || 'GET',
      headers: message.headers || {},
      credentials: message.credentials || 'include',
      mode: message.mode || 'cors'
    };

    if (message.body) {
      fetchOptions.body = message.body;
    }

    // Add timeout
    const controller = new AbortController();
    const timeoutId = setTimeout(() => {
      controller.abort();
      console.warn('⌛ Background: Fetch timeout for', message.url);
    }, message.timeout || 30000);

    fetchOptions.signal = controller.signal;

    console.log('📡 Background: Fetch options:', {
      url: message.url,
      method: fetchOptions.method,
      headersKeys: Object.keys(fetchOptions.headers)
    });

    fetch(message.url, fetchOptions)
      .then(response => {
        clearTimeout(timeoutId);

        // Get response headers
        const responseHeaders = {};
        response.headers.forEach((value, key) => {
          responseHeaders[key] = value;
        });

        console.log('📡 Background: Fetch response:', {
          status: response.status,
          ok: response.ok,
          headers: Object.keys(responseHeaders)
        });

        // Read response body
        return response.text().then(text => ({
          ok: response.ok,
          status: response.status,
          statusText: response.statusText,
          headers: responseHeaders,
          body: text
        }));
      })
      .then(result => {
        console.log('✅ Background: Fetch successful');
        sendResponse({
          success: true,
          ...result
        });
      })
      .catch(error => {
        clearTimeout(timeoutId);
        console.error('❌ Background: Fetch error:', error);
        sendResponse({
          success: false,
          error: error.message || error.toString()
        });
      });

    return true; // Async response
  }

  // Default response for unknown actions
  sendResponse({ status: 'Unknown action' });
  return true;
});
