/**
 * KareeRise Job Extension - API Client
 * 
 * This module handles all server communication for the job extension.
 */

// Server URL configuration - use shared config if available
function getExtensionServerUrl() {
    return window.KareeRiseConfig?.SERVER_URL || "https://kareerise.com";
}

function getBackupUrls() {
    return window.KareeRiseConfig?.BACKUP_URLS || [
        "https://kareerise.com",
        "https://www.kareerise.com",
        "http://127.0.0.1:3000",
        "http://localhost:3000"
    ];
}

console.log(`🔧 Extension configured to use server: ${getExtensionServerUrl()}`);

/**
 * CSP-aware fetch wrapper that routes requests through background script
 * to bypass Content Security Policy restrictions on sites like LinkedIn
 */
async function cspSafeFetch(url, options = {}) {
    console.log(`🔒 [CSP-Safe] Making request to: ${url}`);

    // Try direct fetch first (will work in popup and background contexts)
    try {
        // Quick CSP check - if this throws, we need to use background proxy
        const testFetch = fetch;
        if (typeof testFetch !== 'function') {
            throw new Error('fetch not available');
        }

        // Attempt direct fetch with a very short timeout to detect CSP blocks
        const controller = new AbortController();
        const quickTimeout = setTimeout(() => controller.abort(), 100);

        const directOptions = {
            ...options,
            signal: controller.signal
        };

        const response = await fetch(url, directOptions);
        clearTimeout(quickTimeout);

        console.log(`✅ [CSP-Safe] Direct fetch succeeded`);
        return response;
    } catch (directError) {
        console.log(`🔄 [CSP-Safe] Direct fetch failed (${directError.message}), using background proxy`);

        // Use background script proxy to bypass CSP
        return new Promise((resolve, reject) => {
            const message = {
                action: 'proxyFetch',
                url: url,
                method: options.method || 'GET',
                headers: options.headers || {},
                credentials: options.credentials || 'include',
                mode: options.mode || 'cors',
                body: options.body,
                timeout: options.timeout || 30000
            };

            chrome.runtime.sendMessage(message, (response) => {
                if (chrome.runtime.lastError) {
                    console.error('❌ [CSP-Safe] Background proxy error:', chrome.runtime.lastError);
                    reject(new Error(chrome.runtime.lastError.message));
                    return;
                }

                if (!response) {
                    console.error('❌ [CSP-Safe] No response from background script');
                    reject(new Error('No response from background script'));
                    return;
                }

                if (!response.success) {
                    console.error('❌ [CSP-Safe] Background fetch failed:', response.error);
                    reject(new Error(response.error || 'Background fetch failed'));
                    return;
                }

                console.log(`✅ [CSP-Safe] Background proxy succeeded`);

                // Create a Response-like object
                const mockResponse = {
                    ok: response.ok,
                    status: response.status,
                    statusText: response.statusText,
                    headers: new Map(Object.entries(response.headers || {})),
                    text: () => Promise.resolve(response.body),
                    json: () => {
                        try {
                            return Promise.resolve(JSON.parse(response.body));
                        } catch (e) {
                            return Promise.reject(new Error('Invalid JSON'));
                        }
                    }
                };

                resolve(mockResponse);
            });
        });
    }
}

// Get all server URLs for requests
function getServerUrls() {
    // Basic server URLs - only HTTP for localhost
    const serverUrl = getExtensionServerUrl();
    const backupUrls = getBackupUrls();
    const urls = [serverUrl, ...backupUrls.filter(url => url !== serverUrl)];
    
    // For localhost, only use HTTP - don't add HTTPS variants
    const finalUrls = [];
    urls.forEach(url => {
        if (url.includes('localhost') || url.includes('127.0.0.1')) {
            // Force HTTP for localhost
            const httpUrl = url.replace('https://', 'http://');
            finalUrls.push(httpUrl);
        } else {
            // For production domains, keep as is
            finalUrls.push(url);
        }
    });
    
    // Remove duplicates
    return [...new Set(finalUrls)];
}

/**
 * Submit a job application to the server
 * 
 * @param {Object} jobData - The job data extracted from the page
 * @param {Array} contacts - Optional contacts related to the job
 * @param {Object} authInfo - User authentication information
 * @returns {Promise<Object>} - Server response with job ID
 */
async function submitJobApplication(jobData, contacts = [], authInfo = {}) {
    console.log("🔄 Submitting job application to server...");
    console.log(`🔍 [API DEBUG] Submitting with ${contacts.length} contacts`);
    if (contacts.length > 0) {
        console.log("🔍 [API DEBUG] Contact data being sent:", contacts);
    }
    
    // Format date information in multiple formats to ensure compatibility
    const now = new Date();
    const isoDate = now.toISOString();
    const unixTimestamp = Math.floor(now.getTime() / 1000);
    const formattedDate = `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, "0")}-${String(now.getDate()).padStart(2, "0")}`;
    
    // Get browser cookies for the domain
    let cookieData = { cookies: [] };
    try {
        // Request cookies from background script if extension has cookie permission
        if (chrome.runtime && chrome.runtime.sendMessage) {
            console.log("Requesting cookies from background script");
            
            // Try to get cookies for all possible server URLs
            const cookiePromises = getServerUrls().map(serverUrl => {
                return new Promise((resolve) => {
                    chrome.runtime.sendMessage(
                        { action: "getCookies", url: serverUrl },
                        (response) => {
                            resolve(response || { cookies: [] });
                        }
                    );
                    // Add timeout for each cookie request
                    setTimeout(() => resolve({ cookies: [] }), 1000);
                });
            });
            
            // Wait for all cookie requests to complete
            const cookieResponses = await Promise.all(cookiePromises);
            
            // Merge all cookies, removing duplicates
            const allCookies = [];
            const cookieNames = new Set();
            
            cookieResponses.forEach(response => {
                if (response && response.cookies) {
                    response.cookies.forEach(cookie => {
                        if (!cookieNames.has(cookie.name)) {
                            cookieNames.add(cookie.name);
                            allCookies.push(cookie);
                        }
                    });
                }
            });
            
            cookieData = { cookies: allCookies };
            console.log(`Retrieved ${cookieData.cookies.length} total unique cookies`);
        }
    } catch (e) {
        console.warn("Unable to retrieve cookies:", e);
    }
    
    // Only use authenticated user ID - no development fallbacks
    let effectiveUserId = authInfo.userId;
    
    if (!effectiveUserId || effectiveUserId === 'null' || effectiveUserId === 'undefined') {
        console.log('❌ No authenticated user found');
        return {
            error: "auth_required",
            message: "Please log in to KareeRise before adding jobs",
            redirect: `${getExtensionServerUrl()}/auth/login?returnTo=/applications`
        };
    }
    
    // Prepare the full job payload
    const jobPayload = {
        // Primary data
        title: jobData.title,
        company: jobData.company,
        location: jobData.location,
        description: jobData.description,
        
        // Format variations for compatibility
        jobTitle: jobData.title,
        companyName: jobData.company,
        jobLocation: jobData.location,
        jobDescription: jobData.description,
        
        // URL information
        jobUrl: jobData.jobUrl,
        url: jobData.jobUrl,
        
        // Source tracking
        source: jobData.source || 'unknown',
        jobSource: jobData.source || 'unknown',
        
        // Date in multiple formats
        date: formattedDate,
        dateApplied: formattedDate,
        applicationDate: formattedDate,
        appliedAt: isoDate,
        timestamp: unixTimestamp,
        
        // Browser identification
        browserType: navigator.userAgent.includes("Edg")
            ? "edge"
            : navigator.userAgent.includes("Chrome")
                ? "chrome"
                : navigator.userAgent.includes("Firefox")
                    ? "firefox"
                    : "unknown",
        
        // Extension information
        extensionId: chrome.runtime.id || "unknown",
        
        // Contacts
        contacts: contacts,
        hiringTeamContacts: contacts,
        
        // Authentication information
        userId: effectiveUserId,
        user_id: effectiveUserId, // Alternate format for compatibility 
        userEmail: authInfo.email,
        authenticated: !!effectiveUserId,
        
        // Development info
        isDevelopment: getExtensionServerUrl().includes('localhost')
    };

    // Request options with CORS support
    const requestOptions = {
        method: "POST",
        credentials: "include", // Include cookies for authentication
        referrerPolicy: "no-referrer-when-downgrade",
        mode: "cors",
        headers: {
            "Content-Type": "application/json",
            "X-Extension-Source": "job-importer-extension",
            "X-Extension-Version": "1.2",
            "Accept": "application/json, text/plain, */*",
            "X-Requested-With": "XMLHttpRequest",
            "Origin": window.location.origin
        },
        body: JSON.stringify(jobPayload)
    };
    
    // Add authentication headers if available
    if (effectiveUserId) {
        requestOptions.headers["X-Auth-User-ID"] = effectiveUserId;
        requestOptions.headers["X-User-ID"] = effectiveUserId;
        
        if (authInfo.email) {
            requestOptions.headers["X-Auth-User-Email"] = authInfo.email;
        }
    }
    

    // Add explicit cookies if available
    if (authInfo.cookies && authInfo.cookies.length > 0) {
        const cookieString = authInfo.cookies
            .map((cookie) => `${cookie.name}=${cookie.value}`)
            .join("; ");
            
        requestOptions.headers.Cookie = cookieString;
        requestOptions.headers["X-Auth-Cookies-Count"] = authInfo.cookies.length.toString();
    }

    console.log("🔍 [API] Request options prepared:", {
        serverUrl: getExtensionServerUrl(),
        url: `${getExtensionServerUrl()}/api/extensions/direct-add`,
        method: requestOptions.method,
        mode: requestOptions.mode,
        credentials: requestOptions.credentials,
        headers: Object.keys(requestOptions.headers)
    });

    // Try all endpoints until one succeeds
    const endpoints = [];
    
    // Add direct-add endpoints
    getServerUrls().forEach(url => {
        endpoints.push({
            name: `${url === getExtensionServerUrl() ? 'Primary' : 'Backup'} direct-add`,
            url: `${url}/api/extensions/direct-add`
        });
    });
    
    // Note: Removed job-import fallback as it requires NextAuth session which doesn't work from extension context

    console.log(`🔍 [API] Trying ${endpoints.length} endpoints:`, endpoints.map(e => e.name));
    
    let response = null;
    let responseData = null;
    let lastError = null;
    const timeout = 20000; // 20 seconds
    
    // Try different credentials modes for each endpoint
    const credentialsModes = ["include", "same-origin", "omit"];
    
    // Try each endpoint with timeout and different credentials modes
    for (const endpoint of endpoints) {
        for (const credentialsMode of credentialsModes) {
            try {
                console.log(`🔍 [API] Trying ${endpoint.name} with credentials mode: ${credentialsMode}...`);
                console.log(`🔍 [API] Full URL: ${endpoint.url}`);
                console.log(`🔍 [API] Request headers:`, Object.keys(requestOptions.headers));
                
                const controller = new AbortController();
                const timeoutId = setTimeout(() => controller.abort(), timeout);
                
                // Create a copy of requestOptions with current credentials mode
                const currentOptions = {
                    ...requestOptions,
                    credentials: credentialsMode,
                    signal: controller.signal
                };
                
                response = await cspSafeFetch(endpoint.url, currentOptions);

                clearTimeout(timeoutId);
                
                console.log(`🔍 [API] Response from ${endpoint.name} (${credentialsMode}):`, {
                    status: response.status,
                    ok: response.ok,
                    statusText: response.statusText
                });
                
                if (response.ok) {
                    responseData = await response.json();
                    console.log(`✅ [API] Success response from ${endpoint.name} (${credentialsMode}):`, responseData);
                    return responseData;
                } else if (response.status === 401 || response.status === 403) {
                    // Handle authentication errors
                    const authError = await response.json().catch(() => ({
                        error: "Authentication required",
                        redirect: `${getExtensionServerUrl()}/auth/login?returnTo=/applications`
                    }));
                    
                    console.warn(`🔒 [API] Authentication required from ${endpoint.name} (${credentialsMode}):`, authError);
                    
                    // Try the next credentials mode or endpoint before returning auth error
                    continue;
                }
            } catch (error) {
                lastError = error;
                console.error(`❌ [API] Error with ${endpoint.name} (${credentialsMode}):`, error.message);
                
                // Continue to the next credentials mode or endpoint
            }
        }
        
        // Brief delay before trying next endpoint
        await new Promise(resolve => setTimeout(resolve, 300));
    }
    
    // If we've tried all endpoints and none succeeded
    console.error("❌ [API] All endpoints and credentials modes failed:", lastError);
    
    // No fallback - if authentication fails, user must log in
    console.log("❌ [API] All authenticated requests failed - user must log in");
    
    // If we get here, all attempts failed
    return {
        error: "server_error",
        message: lastError?.message || "Failed to communicate with server",
        details: lastError
    };
}

/**
 * Submit job data for AI extraction
 * 
 * @param {Object} htmlData - HTML content of the job page
 * @param {Object} authInfo - User authentication information
 * @returns {Promise<Object>} - Server response with extracted job data
 */
async function submitJobForAIExtraction(htmlData, authInfo = {}) {
    console.log("🤖 [API] Submitting job for AI extraction (GPT will extract contacts from HTML)...");
    
    // Prepare payload - for universal AI extraction, we send the full HTML
    // Maximum size increased since we're now sending universal content
    const MAX_HTML_SIZE = 100000; // 100KB should be sufficient for most job pages
    
    // Get the full HTML content
    let fullHTML = htmlData.fullHTML || '';
    
    if (fullHTML.length > MAX_HTML_SIZE) {
        console.warn(`⚠️ [API] Full HTML is too large (${fullHTML.length} chars), truncating to ${MAX_HTML_SIZE} chars`);
        fullHTML = fullHTML.substring(0, MAX_HTML_SIZE) + '\n<!-- [TRUNCATED BY API CLIENT] -->';
    }
    
    console.log(`🔍 [API] Sending ${fullHTML.length} characters of HTML to AI for processing`);
    
    // Only use authenticated user ID - no development fallbacks
    let effectiveUserId = authInfo.userId;
    
    if (!effectiveUserId || effectiveUserId === 'null' || effectiveUserId === 'undefined') {
        console.log('❌ No authenticated user found for AI extraction');
        return {
            error: "auth_required",
            message: "Please log in to KareeRise before using AI extraction",
            redirect: `${getExtensionServerUrl()}/auth/login?returnTo=/applications`
        };
    }
    
    // Create the payload with the full HTML content for universal processing
    const payload = {
        pageTitle: htmlData.title || '',
        pageUrl: htmlData.url || '',
        htmlContent: fullHTML,
        source: htmlData.source || 'unknown',  // Include detected source
        // Add authentication information
        userId: effectiveUserId,
        user_id: effectiveUserId,
        userEmail: authInfo.email,
        // Add a timestamp to prevent caching issues
        timestamp: Date.now()
    };
    
    console.log(`🔍 [API] Sending payload with ${fullHTML.length} characters of HTML content`);
    
    // Try different credentials modes for each endpoint
    const credentialsModes = ["include", "same-origin", "omit"];
    const timeout = 30000; // 30 seconds per endpoint (increased for GPT processing)
    
    // Try all endpoints and credentials modes
    for (const credentialsMode of credentialsModes) {
        // Try all server URLs
        for (const serverUrl of getServerUrls()) {
            const endpoint = `${serverUrl}/api/extensions/ai-extract`;
            
            console.log(`🔍 [API] Trying AI extraction with endpoint: ${endpoint} (${credentialsMode})`);
            
            try {
                // Configure request options for this attempt
                const requestOptions = {
                    method: "POST",
                    credentials: credentialsMode,
                    mode: "cors",
                    headers: {
                        "Content-Type": "application/json",
                        "X-Extension-Source": "job-importer-extension",
                        "X-Extension-Version": "1.2",
                        "Accept": "application/json, text/plain, */*",
                        "X-Requested-With": "XMLHttpRequest",
                        "Origin": window.location.origin
                    },
                    body: JSON.stringify(payload)
                };
                
                // Add auth headers
                if (effectiveUserId) {
                    requestOptions.headers["X-Auth-User-ID"] = effectiveUserId;
                    requestOptions.headers["X-User-ID"] = effectiveUserId;
                }
                
                
                // Add timeout to prevent hanging requests
                const controller = new AbortController();
                const timeoutId = setTimeout(() => {
                    controller.abort();
                    console.warn(`⌛ [API] Timeout for endpoint ${endpoint}`);
                }, timeout);
                
                requestOptions.signal = controller.signal;

                // Make the fetch request using CSP-safe wrapper
                const response = await cspSafeFetch(endpoint, requestOptions);
                clearTimeout(timeoutId);
                
                console.log(`🔍 [API] Response from ${endpoint}:`, {
                    status: response.status,
                    ok: response.ok
                });
                
                if (response.ok) {
                    const data = await response.json();
                    console.log("✅ [API] AI extraction successful:", data);
                    return data;
                }
                
                // Try to get error details
                try {
                    const errorText = await response.text();
                    console.warn(`⚠️ [API] Error details: ${errorText.substring(0, 200)}`);
                } catch (e) {
                    console.warn("Could not extract error text");
                }
                
                // If auth error, don't keep trying other credentials modes
                if (response.status === 401 || response.status === 403) {
                    console.warn("🔒 [API] Authentication required for AI extraction");
                    
                    return {
                        error: "auth_required",
                        message: "Please log in to KareeRise before using AI extraction",
                        redirect: `${getExtensionServerUrl()}/auth/login?returnTo=/applications`
                    };
                }
            } catch (endpointError) {
                console.error(`❌ [API] Error with endpoint ${endpoint}:`, endpointError);
            }
            
            // Brief delay before trying next URL
            await new Promise(resolve => setTimeout(resolve, 300));
        }
    }
    
    // If all regular attempts failed, try a minimal request as last resort
    console.log("🔄 [API] All standard attempts failed, trying minimal request...");
    
    try {
        // Create a minimal payload with just the essential information
        const minimalPayload = {
            pageUrl: htmlData.url || window.location.href,
            pageTitle: htmlData.title || document.title,
            // Skip all HTML content
            userId: effectiveUserId,
            user_id: effectiveUserId,
            // Flag to indicate this is a minimal request
            minimalRequest: true,
            timestamp: Date.now()
        };
        
        // Try all server URLs with minimal payload
        for (const serverUrl of getServerUrls()) {
            const endpoint = `${serverUrl}/api/extensions/ai-extract`;
            
            console.log(`🔍 [API] Trying minimal AI extraction with endpoint: ${endpoint}`);
            
            try {
                const minimalOptions = {
                    method: "POST",
                    credentials: "include",
                    mode: "cors",
                    headers: {
                        "Content-Type": "application/json",
                        "X-Extension-Source": "job-importer-extension",
                        "X-Auth-User-ID": effectiveUserId,
                        "X-User-ID": effectiveUserId,
                        "Origin": window.location.origin
                    },
                    body: JSON.stringify(minimalPayload)
                };
                
                // Add timeout
                const controller = new AbortController();
                const timeoutId = setTimeout(() => controller.abort(), 8000);
                minimalOptions.signal = controller.signal;

                // Make the request using CSP-safe wrapper
                const response = await cspSafeFetch(endpoint, minimalOptions);
                clearTimeout(timeoutId);
                
                if (response.ok) {
                    const data = await response.json();
                    console.log("✅ [API] Minimal AI extraction successful:", data);
                    return data;
                }
            } catch (minimalError) {
                console.error(`❌ [API] Error with minimal request to ${endpoint}:`, minimalError);
            }
        }
        
        // If we get here, all attempts failed
        throw new Error("All AI extraction endpoints failed");
    } catch (error) {
        console.error("❌ [API] AI extraction request failed:", error);
        
        return {
            error: "network_error",
            message: "Failed to communicate with AI extraction service",
            details: error.message,
            suggestion: "Please try using the standard 'Add to KareeRise' button instead"
        };
    }
}

/**
 * Get current user information
 * 
 * @returns {Promise<Object>} - User information
 */
async function getCurrentUser() {
    console.log("🔍 [API] Getting current user information...");
    console.log("🔍 [API] Current page URL:", window.location.href);
    console.log("🔍 [API] Document cookies available:", document.cookie ? "Yes" : "No");
    console.log("🔍 [API] Document cookies:", document.cookie);
    
    // Use background script to get user info (has access to localhost cookies)
    try {
        console.log("🔄 [API] Requesting user info from background script...");
        
        const backgroundResponse = await new Promise((resolve) => {
            chrome.runtime.sendMessage(
                { action: "getCurrentUser", url: getExtensionServerUrl() },
                (response) => resolve(response)
            );
        });
        
        console.log("🔍 [API] Background script response:", backgroundResponse);
        
        if (backgroundResponse && backgroundResponse.authenticated) {
            console.log("✅ [API] Background script authenticated user successfully");
            return {
                userId: backgroundResponse.userId,
                email: backgroundResponse.email,
                authenticated: true,
                method: 'background'
            };
        }
        
        console.log("⚠️ [API] Background script couldn't authenticate, trying direct request...");
    } catch (backgroundError) {
        console.error("❌ [API] Background script error:", backgroundError);
        console.log("⚠️ [API] Falling back to direct request...");
    }
    
    // Test both endpoints
    const authUrls = getServerUrls().map(url => `${url}/api/auth/session`);
    const testUrls = getServerUrls().map(url => `${url}/api/test-extension`);
    const urls = [...testUrls, ...authUrls];
    
    console.log("🔍 [API] Will try these URLs for authentication:", urls);
    
    // Try different credentials modes
    const credentialsModes = ["include", "same-origin", "omit"];
    
    // Try each endpoint with different credentials modes
    for (const credentialsMode of credentialsModes) {
        // Try each URL
        for (const url of urls) {
            try {
                console.log(`🔍 [API] Trying to get user from ${url} with credentials: ${credentialsMode}`);
                
                // Configure request options
                const requestOptions = {
                    method: "GET",
                    credentials: credentialsMode,
                    headers: {
                        "Accept": "application/json",
                        "X-Extension-Source": "job-importer-extension",
                        "X-Extension-Version": "1.2"
                    }
                };
                
                // Don't set Origin header for localhost requests to avoid CORS issues
                if (!url.includes('localhost')) {
                    requestOptions.headers["Origin"] = window.location.origin;
                }
                
                console.log(`🔍 [API] Request to ${url} with credentials: ${credentialsMode}`);
                
                
                // Add timeout
                const controller = new AbortController();
                const timeoutId = setTimeout(() => controller.abort(), 5000);
                requestOptions.signal = controller.signal;

                // Make the request using CSP-safe wrapper
                const response = await cspSafeFetch(url, requestOptions);
                clearTimeout(timeoutId);
                
                console.log(`🔍 [API] Response from ${url}:`, {
                    status: response.status,
                    ok: response.ok
                });
                
                if (response.ok) {
                    const data = await response.json();
                    console.log("✅ [API] Response:", data);
                    console.log("✅ [API] Response from:", url);
                    console.log("✅ [API] Response type:", typeof data);
                    console.log("✅ [API] Response keys:", Object.keys(data || {}));
                    
                    // Handle test endpoint response
                    if (url.includes('/api/test-extension')) {
                        console.log("🔬 [API] Test endpoint response:", data);
                        if (data.hasSession && data.session && data.session.user) {
                            console.log("✅ [API] Test endpoint found valid session!");
                            return {
                                userId: data.session.user.id,
                                email: data.session.user.email,
                                authenticated: true,
                                url: url,
                                testEndpoint: true
                            };
                        }
                        // Continue to try other endpoints if test fails
                        continue;
                    }
                    
                    // Handle NextAuth session endpoint response
                    if (data && data.user && data.user.id) {
                        console.log("✅ [API] Successfully authenticated:", {
                            userId: data.user.id,
                            email: data.user.email
                        });
                        
                        return {
                            userId: data.user.id,
                            email: data.user.email,
                            authenticated: true,
                            url: url // Include the successful URL for debugging
                        };
                    } else {
                        console.log("❌ [API] No active session found in this response");
                        // Continue to try other endpoints
                    }
                }
            } catch (error) {
                console.error(`❌ [API] Error with ${url} (${credentialsMode}):`, error.message);
            }
            
            // Brief delay before trying next URL
            await new Promise(resolve => setTimeout(resolve, 200));
        }
    }
    
    
    console.warn("❌ [API] No successful authentication found");
    
    return {
        authenticated: false,
        error: "Not authenticated",
        suggestion: "Please log in to KareeRise before using the extension"
    };
}

/**
 * Open login page for authentication
 * @param {string} returnUrl - The URL to return to after login
 */
function openLoginPage(returnUrl = '/applications') {
    // Ensure we're using the correct localhost URL
    const baseUrl = getExtensionServerUrl() || 'https://kareerise.com';
    const loginUrl = `${baseUrl}/auth/login?returnTo=${encodeURIComponent(returnUrl)}`;
    
    console.log(`🔒 [API] Opening login page: ${loginUrl}`);
    console.log(`🔒 [API] Base URL: ${baseUrl}`);
    console.log(`🔒 [API] Full URL being opened: ${loginUrl}`);
    
    // Try multiple methods to ensure the URL opens correctly
    try {
        // Method 1: Direct window.open
        const newWindow = window.open(loginUrl, '_blank', 'width=800,height=600,scrollbars=yes,resizable=yes');
        
        if (newWindow) {
            // Focus the new window
            newWindow.focus();
            console.log('✅ Login window opened successfully');
        } else {
            // Fallback: try without window features
            console.warn('⚠️ Popup blocked, trying without window features');
            window.open(loginUrl, '_blank');
        }
    } catch (error) {
        console.error('❌ Error opening login window:', error);
        // Last resort: show URL to user
        alert(`Please manually open this URL to login:\n${loginUrl}`);
    }
}

/**
 * Test the AI extraction endpoint connectivity
 * @returns {Promise<Object>} The test result
 */
async function testAIExtraction() {
    console.log("🔧 [API] Testing AI extraction endpoint...");
    
    // Create a minimal test payload
    const testPayload = {
        pageUrl: "https://example.com/test-job",
        pageTitle: "Test Job Title",
        htmlContent: "<div>Test job description</div>"
    };
    
    // First, check if server is reachable at all with a simple GET request
    try {
        const serverCheckUrl = `${getExtensionServerUrl()}/api/user/current`;
        console.log(`🔍 [API] Checking if server is reachable at ${serverCheckUrl}`);
        
        const serverCheckResponse = await cspSafeFetch(serverCheckUrl, {
            method: "GET",
            mode: "cors",
            headers: {
                "X-Extension-Source": "job-importer-extension-test"
            },
            credentials: "same-origin"
        }).catch(error => {
            console.error("❌ Server unreachable:", error.message);
            return null;
        });
        
        if (!serverCheckResponse) {
            return {
                success: false,
                error: "Server unreachable - please check your KareeRise connection",
                serverUrl: getExtensionServerUrl(),
                suggestions: [
                    "Make sure the server is running",
                    "Check the port number (3000)",
                    "Verify the server accepts requests from the extension"
                ]
            };
        }
        
        console.log(`🔍 Server response status: ${serverCheckResponse.status}`);
    } catch (serverCheckError) {
        console.error("❌ Error checking server:", serverCheckError);
    }
    
    // Prepare different request options to try
    const requestVariations = [
        // Variation 1: Standard CORS with credentials
        {
            method: "POST",
            mode: "cors",
            credentials: "include",
            headers: {
                "Content-Type": "application/json",
                "X-Extension-Source": "job-importer-extension-test",
                "Origin": window.location.origin
            },
            body: JSON.stringify(testPayload)
        },
        // Variation 2: CORS without credentials
        {
            method: "POST",
            mode: "cors",
            headers: {
                "Content-Type": "application/json",
                "X-Extension-Source": "job-importer-extension-test",
                "Origin": window.location.origin
            },
            body: JSON.stringify(testPayload)
        },
        // Variation 3: same-origin (for localhost)
        {
            method: "POST",
            mode: "cors",
            credentials: "same-origin",
            headers: {
                "Content-Type": "application/json",
                "X-Extension-Source": "job-importer-extension-test",
                "Origin": window.location.origin
            },
            body: JSON.stringify(testPayload)
        }
    ];
    
    // Store all test results
    const results = {
        attemptedEndpoints: [],
        errors: []
    };
    
    try {
        // Try all server URLs with different request variations
        for (const url of getServerUrls()) {
            const endpoint = `${url}/api/extensions/ai-extract`;
            console.log(`🔍 [API] Testing AI extraction with endpoint: ${endpoint}`);
            
            for (const [index, options] of requestVariations.entries()) {
                try {
                    console.log(`🔍 [API] Trying variation ${index+1} with ${endpoint}`);
                    
                    // Add timeout to prevent hanging requests
                    const controller = new AbortController();
                    const timeoutId = setTimeout(() => {
                        controller.abort();
                        console.log(`⌛ Request timeout for ${endpoint} variation ${index+1}`);
                    }, 5000);
                    
                    // Check OPTIONS preflight first to verify CORS
                    try {
                        const preflightResponse = await cspSafeFetch(endpoint, {
                            method: "OPTIONS",
                            headers: {
                                "Origin": window.location.origin,
                                "Access-Control-Request-Method": "POST"
                            },
                            signal: controller.signal
                        });
                        
                        const preflightHeaders = {};
                        preflightResponse.headers.forEach((value, key) => {
                            preflightHeaders[key] = value;
                        });
                        
                        console.log(`🔍 OPTIONS preflight for ${endpoint}:`, {
                            status: preflightResponse.status,
                            headers: preflightHeaders
                        });
                    } catch (preflightError) {
                        console.warn(`⚠️ OPTIONS preflight failed:`, preflightError.message);
                    }
                    
                    // Make the actual POST request using CSP-safe wrapper
                    const response = await cspSafeFetch(endpoint, {
                        ...options,
                        signal: controller.signal
                    });
                    
                    clearTimeout(timeoutId);
                    
                    // Log response headers for debugging
                    const responseHeaders = {};
                    response.headers.forEach((value, key) => {
                        responseHeaders[key] = value;
                    });
                    
                    console.log(`🔍 Response from ${endpoint}:`, {
                        status: response.status,
                        statusText: response.statusText,
                        headers: responseHeaders
                    });
                    
                    // Store the attempted endpoint
                    results.attemptedEndpoints.push({
                        url: endpoint,
                        variation: index+1,
                        status: response.status,
                        statusText: response.statusText
                    });
                    
                    if (response.ok) {
                        console.log(`✅ [API] AI extraction test successful with ${endpoint} (variation ${index+1})`);
                        try {
                            const data = await response.json();
                            return {
                                success: true,
                                endpoint,
                                variation: index+1,
                                status: response.status,
                                data,
                                headers: responseHeaders
                            };
                        } catch (jsonError) {
                            console.error("Error parsing JSON response:", jsonError);
                            
                            // Try to get text response
                            const textResponse = await response.text();
                            console.log("Raw response:", textResponse.substring(0, 500));
                            
                            // Still return success since the request worked
                            return {
                                success: true,
                                endpoint,
                                variation: index+1,
                                status: response.status,
                                parseError: jsonError.message,
                                rawResponse: textResponse.substring(0, 500)
                            };
                        }
                    } else {
                        // Try to read error response
                        try {
                            const errorText = await response.text();
                            console.warn(`⚠️ Error response: ${errorText.substring(0, 200)}`);
                            results.errors.push({
                                endpoint,
                                variation: index+1,
                                status: response.status,
                                error: errorText.substring(0, 200)
                            });
                        } catch (e) {
                            console.warn(`⚠️ Could not read error response`);
                        }
                    }
                } catch (error) {
                    console.error(`❌ [API] Error with endpoint ${endpoint} (variation ${index+1}):`, error);
                    results.errors.push({
                        endpoint,
                        variation: index+1,
                        error: error.message
                    });
                }
            }
        }
        
        // If we get here, all attempts failed
        console.error("❌ All AI extraction tests failed");
        return {
            success: false,
            error: "All AI extraction endpoints and variations failed",
            details: results,
            suggestions: [
                "Check if server is running",
                "Verify the endpoint URL is correct",
                "Check browser console for CORS errors",
                "Make sure the AI extraction API is implemented on the server"
            ]
        };
    } catch (error) {
        console.error("❌ [API] Error testing AI extraction:", error);
        return {
            success: false,
            error: error.message,
            details: results
        };
    }
}

/**
 * Test AI extraction with minimal payload
 * This helps diagnose if the issue is with payload size
 */
async function testMinimalAIExtraction() {
    console.log("🔧 [API] Testing AI extraction with minimal payload...");
    
    // Create the minimal payload
    const minimalPayload = {
        pageUrl: "https://example.com/test-minimal",
        pageTitle: "Test Minimal Job",
        // No HTML content at all
    };
    
    // Create minimal request options
    const minimalOptions = {
        method: "POST",
        mode: "cors",
        credentials: "include",
        headers: {
            "Content-Type": "application/json",
            "X-Extension-Source": "job-importer-extension-minimal",
            "X-Extension-Version": "1.2",
            "X-Extension-Source": "job-importer-extension-minimal",
            "Origin": window.location.origin
        },
        body: JSON.stringify(minimalPayload)
    };
    
    try {
        // Try all server URLs
        for (const url of getServerUrls()) {
            const endpoint = `${url}/api/extensions/ai-extract`;
            console.log(`🔍 [API] Testing minimal AI extraction with: ${endpoint}`);
            
            try {
                // Add timeout
                const controller = new AbortController();
                const timeoutId = setTimeout(() => controller.abort(), 5000);

                // Try the request using CSP-safe wrapper
                const response = await cspSafeFetch(endpoint, {
                    ...minimalOptions,
                    signal: controller.signal
                });
                
                clearTimeout(timeoutId);
                
                console.log(`🔍 Minimal test response from ${endpoint}:`, {
                    status: response.status,
                    ok: response.ok
                });
                
                if (response.ok) {
                    console.log(`✅ [API] Minimal AI extraction test successful with ${endpoint}`);
                    try {
                        const data = await response.json();
                        return {
                            success: true,
                            endpoint,
                            status: response.status,
                            data
                        };
                    } catch (jsonError) {
                        const textResponse = await response.text();
                        return {
                            success: true,
                            endpoint,
                            status: response.status,
                            rawResponse: textResponse.substring(0, 200)
                        };
                    }
                }
            } catch (error) {
                console.error(`❌ [API] Error with minimal test for ${endpoint}:`, error);
            }
        }
        
        return {
            success: false,
            error: "All minimal AI extraction tests failed",
            suggestions: [
                "This suggests the issue is not related to payload size",
                "Check CORS configuration and credentials mode"
            ]
        };
    } catch (error) {
        console.error("❌ [API] Error testing minimal AI extraction:", error);
        return {
            success: false,
            error: error.message
        };
    }
}

// Export the API client
window.KareeRiseAPIClient = {
    getExtensionServerUrl,
    submitJobApplication,
    submitJobForAIExtraction,
    getCurrentUser,
    openLoginPage,
    testAIExtraction,
    testMinimalAIExtraction
};
